package edu.uky.ai.planning.ss;

import edu.uky.ai.logic.State;
import edu.uky.ai.planning.ForwardPlan;
import edu.uky.ai.planning.Plan;
import edu.uky.ai.planning.Step;

/**
 * Represents a node in a forward state space search graph, where a node is a
 * state and the plan taken to reach that state from the initial state.
 * 
 * @author Stephen G. Ware
 */
public class ForwardNode implements StateSpaceNode {

	/** The plan taken to reach this state */
	public final ForwardPlan plan;
	
	/** The current state */
	public final State state;
	
	/** This node's parent node (i.e. the state before the last step) */
	public final ForwardNode parent;
	
	/** Indicates whether or not this node has been visited */
	private boolean visited = false;
	
	/**
	 * Constructs a new root node with the given initial state.
	 * 
	 * @param initial the problem's initial state
	 */
	ForwardNode(State initial) {
		this.plan = new ForwardPlan();
		this.state = initial;
		this.parent = null;
	}
	
	/**
	 * Constructs a new node with a given parent and most recent step.
	 * 
	 * @param parent the previous state
	 * @param step the step to take next
	 * @param state the state that results from taking that next step
	 */
	private ForwardNode(ForwardNode parent, Step step, State state) {
		this.plan = parent.plan.addStep(step);
		this.state = state;
		this.parent = parent;
		ForwardRoot root = getRoot();
		if(!parent.visited) {
			root.budget.incrementOperations();
			root.budget.checkTime();
			parent.visited = true;
			root.visited++;
		}
		root.generated++;
	}
	
	@Override
	public String toString() {
		String str = "===== FORWARD NODE =====";
		str += "\n Plan: ";
		boolean first = true;
		for(Step step : plan) {
			if(first)
				first = false;
			else
				str += "\n       ";
			str += step;
		}
		str += "\n State: " + state;
		return str;
	}
	
	@Override
	public Plan getPlan() {
		return plan;
	}
	
	@Override
	public final ForwardRoot getRoot() {
		ForwardNode current = this;
		while(current.parent != null)
			current = current.parent;
		return (ForwardRoot) current;
	}
	
	/**
	 * Expands the child node that would be generated by taking this step.  If
	 * the parent has not yet been visited, the parent is marked as visited.
	 * 
	 * @param step the step which results in the child state
	 * @param state the child state
	 * @return a node for the resulting state
	 */
	public ForwardNode expand(Step step, State state) {
		return new ForwardNode(this, step, state);
	}
}
