package edu.uky.ai.planning.ss;

import edu.uky.ai.logic.Conjunction;
import edu.uky.ai.logic.Literal;
import edu.uky.ai.logic.Proposition;
import edu.uky.ai.planning.BackwardPlan;
import edu.uky.ai.planning.Plan;
import edu.uky.ai.planning.Step;

/**
 * Represents a node in a backward state space search graph, where a node is a
 * goal and a plan to achieve the problem goal if the node's goal can be
 * achieved.
 * 
 * @author Stephen G. Ware
 */
public class BackwardNode implements StateSpaceNode {
	
	/** The goal to be achieve that will allow the plan to succeed */
	public final Conjunction goal;

	/** The plan to reach the goal */
	public final BackwardPlan plan;
	
	/** This node's parent node (i.e. the rest of the plan after the first step) */
	public final BackwardNode parent;
	
	/** Indicates whether or not this node has been visited */
	private boolean visited = false;
	
	/**
	 * Constructs a new root node with the given problem goal.
	 * 
	 * @param initial the problem's goal
	 */
	BackwardNode(Proposition goal) {
		if(goal instanceof Conjunction)
			this.goal = (Conjunction) goal;
		else
			this.goal = new Conjunction((Literal) goal);
		this.plan = new BackwardPlan();
		this.parent = null;
	}
	
	/**
	 * Constructs a new node with a given parent and most recent goal.
	 * 
	 * @param parent the previous goal
	 * @param goal the goal that must be achieved to execute the step
	 * @param step the first step to take
	 */
	private BackwardNode(BackwardNode parent, Conjunction goal, Step step) {
		this.goal = goal;
		this.plan = parent.plan.addStep(step);
		this.parent = parent;
		BackwardRoot root = getRoot();
		if(!parent.visited) {
			root.budget.incrementOperations();
			root.budget.checkTime();
			parent.visited = true;
			root.visited++;
		}
		root.generated++;
	}
	
	@Override
	public String toString() {
		String str = "===== BACKWARD NODE =====";
		str += "\n Goal: " + goal;
		str += "\n Plan: ";
		boolean first = true;
		for(Step step : plan) {
			if(first)
				first = false;
			else
				str += "\n       ";
			str += step;
		}
		return str;
	}
	
	@Override
	public Plan getPlan() {
		return plan;
	}
	
	@Override
	public final BackwardRoot getRoot() {
		BackwardNode current = this;
		while(current.parent != null)
			current = current.parent;
		return (BackwardRoot) current;
	}
	
	/**
	 * Expands the child node that would be generated if the given step were
	 * the first one to be taken. If the parent has not yet been visited, the
	 * parent is marked as visited.
	 * 
	 * @param goal the goal that must be achieved to take the given step
	 * @param step the first step to take
	 * @return a node for the resulting goal
	 */
	public BackwardNode expand(Conjunction goal, Step step) {
		return new BackwardNode(this, goal, step);
	}
}
