package edu.uky.ai.planning.ps;

import edu.uky.ai.logic.HashSubstitution;
import edu.uky.ai.logic.ListBindings;
import edu.uky.ai.logic.Literal;
import edu.uky.ai.logic.Proposition;
import edu.uky.ai.logic.Substitution;
import edu.uky.ai.logic.Variable;
import edu.uky.ai.planning.Operator;
import edu.uky.ai.planning.Utilities;
import edu.uky.ai.util.ImmutableArray;

/**
 * This class represents a step in a partial plan.  A partial step is like an
 * {@link edu.uky.ai.planning.Operator Operator} because it has parameters, and
 * like a {@link edu.uky.ai.planning.Step Step} because those parameters will
 * eventually all be ground in specific values from the bindings of a partial
 * plan.
 * 
 * @author Stephen G. Ware
 */
public class PartialStep implements Partial {

	/** An empty set of parameters */
	private static final ImmutableArray<Variable> NO_PARAMETERS = new ImmutableArray<>(new Variable[0]);
	
	/** An empty set of literals */
	private static final ImmutableArray<Literal> NO_LITERALS = new ImmutableArray<>(new Literal[0]);
	
	/** The operator of which this step is an instance (or null if this step is not an instance of any operator) */
	public final Operator operator;
	
	/** All the variables used in this step */
	public final ImmutableArray<Variable> parameters;
	
	/** An array of this step's preconditions */
	public final ImmutableArray<Literal> preconditions;
	
	/** An array of this step's effects */
	public final ImmutableArray<Literal> effects;
	
	/**
	 * Constructs a new partial step with a given precondition and effect.
	 * 
	 * @param precondition the step's precondition
	 * @param effect the step's effect
	 */
	PartialStep(Proposition precondition, Proposition effect) {
		this.operator = null;
		this.parameters = NO_PARAMETERS;
		this.preconditions = precondition == Proposition.TRUE ? NO_LITERALS : new ImmutableArray<>(Utilities.toLiterals(precondition));
		this.effects = effect == Proposition.TRUE ? NO_LITERALS : new ImmutableArray<>(Utilities.toLiterals(effect));
	}
	
	/**
	 * Constructs a partial step from an
	 * {@link edu.uky.ai.planning.Operator Operator} (a step template).
	 * 
	 * @param operator the operator of which this step will be an instance
	 */
	public PartialStep(Operator operator) {
		this.operator = operator;
		HashSubstitution substitution = new HashSubstitution();
		Variable[] parameters = new Variable[operator.parameters.size()];
		for(int i=0; i<parameters.length; i++) {
			parameters[i] = operator.parameters.get(i).makeUnique();
			substitution.set(operator.parameters.get(i), parameters[i]);
		}
		this.parameters = new ImmutableArray<>(parameters);
		this.preconditions = new ImmutableArray<Literal>(Utilities.toLiterals(operator.precondition.substitute(substitution)));
		this.effects = new ImmutableArray<Literal>(Utilities.toLiterals(operator.effect.substitute(substitution)));
	}
	
	@Override
	public String toString() {
		return toString(ListBindings.EMPTY);
	}
	
	@Override
	public String toString(Substitution substitution) {
		if(isStart())
			return "start";
		else if(isEnd())
			return "end";
		String str = "(" + operator.name;
		for(Variable parameter : parameters)
			str += " " + parameter.substitute(substitution);
		str += ")";
		return str;
	}
	
	/**
	 * Tests whether this step is a dummy start step.
	 * 
	 * @return true is this is a dummy start step, false otherwise
	 */
	private final boolean isStart() {
		return operator == null && preconditions == NO_LITERALS;
	}
	
	/**
	 * Tests whether this step is a dummy end step.
	 * 
	 * @return true is this is a dummy end step, false otherwise
	 */
	private final boolean isEnd() {
		return operator == null && effects == NO_LITERALS;
	}
	
	/**
	 * Given a substitution that specific values for all of this step's
	 * parameters, this method will return the
	 * {@link edu.uky.ai.planning.Step Step} object that this object
	 * represents.
	 * 
	 * @param substitution a substitution containing values for all of this step's parameters
	 * @return a step
	 */
	public edu.uky.ai.planning.Step makeStep(Substitution substitution) {
		HashSubstitution mapping = new HashSubstitution();
		for(int i=0; i<parameters.size(); i++)
			mapping.set(operator.parameters.get(i), substitution.get(parameters.get(i)));
		return operator.makeStep(mapping);
	}
}
