package edu.uky.ai.planning.pg;

import edu.uky.ai.SearchBudget;
import edu.uky.ai.planning.Plan;
import edu.uky.ai.planning.Search;

/**
 * This class represents the second phase of a {@link PlanGraphSearch}, in
 * which we search for a subgraph of a plan graph that represents a solution
 * to the problem.  The members of the plan graph family of planners vary in
 * how they implement this phase of the search.
 * 
 * @author Stephen G. Ware
 */
public abstract class SubgraphSearch extends Search {

	/** The root node of the search space */
	protected final SubgraphSpaceRoot root;
	
	/**
	 * Constructs a new subgraph space search process for the given plan graph
	 * and starting with the given subset of nodes from that graph.
	 * 
	 * @param graph the plan graph to be searched
	 * @param budget the search budget
	 * @param nodes a subset of nodes from the plan graph
	 */
	public SubgraphSearch(PlanGraph graph, SearchBudget budget, Subgraph nodes) {
		super(graph.problem, budget);
		this.root = new SubgraphSpaceRoot(graph, budget, nodes);
	}

	@Override
	public int countVisited() {
		return root.visited;
	}

	@Override
	public int countGenerated() {
		return root.generated;
	}
	
	@Override
	public final Plan solve() {
		SubgraphSpaceNode solution = findSubgraph();
		if(solution == null)
			return null;
		else
			return solution.subgraph.getPlan();
	}
	
	/**
	 * Searches the space of {@link SubgraphSpaceNode}s until a node which
	 * represents a solution is found.
	 * 
	 * @return the node representing a solution
	 */
	protected abstract SubgraphSpaceNode findSubgraph();
}
