package edu.uky.ai.planning.pg;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.NoSuchElementException;

import edu.uky.ai.planning.ForwardPlan;
import edu.uky.ai.planning.Plan;

/**
 * Represents a specific subset of nodes in a plan graph.  Subsets are
 * generally used when searching a plan graph for a solution plan.  This is an
 * immutable data structure, so its {@link #add(Node, int) add} and
 * {@link #remove(Node, int) remove} methods do not modify this object, but
 * instead return a new subgraph that reflects the changes.  Note that
 * this is a set, meaning that it will not contain duplicates.
 * 
 * @author Stephen G. Ware
 */
public class Subgraph implements Iterable<NodeInstance> {

	/** The first node (and the level at which it appears) in the subgraph */
	public NodeInstance first;
	
	/** The rest of the subgraph */
	public Subgraph rest;
	
	/**
	 * Constructs a new subgraph with the given first node and rest subgraph.
	 * 
	 * @param first the first node in this subgraph
	 * @param rest the rest of the subgraph
	 */
	private Subgraph(NodeInstance first, Subgraph rest) {
		this.first = first;
		this.rest = rest;
	}
	
	/**
	 * Constructs an empty subgraph, containing no nodes.
	 */
	public Subgraph() {
		this(null, null);
	}
	
	@Override
	public boolean equals(Object other) {
		if(other instanceof Subgraph) {
			Subgraph otherSet = (Subgraph) other;
			if(size() == otherSet.size()) {
				if(size() == 0)
					return true;
				else if(first == otherSet.first)
					return rest.equals(otherSet.rest);
			}
		}
		return false;
	}
	
	@Override
	public int hashCode() {
		if(size() == 0)
			return 0;
		else
			return first.hashCode() + rest.hashCode();
	}
	
	@Override
	public String toString() {
		if(first == null)
			return "(empty)";
		PlanGraph graph = first.node.graph;
		String string = "";
		for(int i=0; i<graph.size(); i++) {
			string += "\nLevel " + i + ":";
			for(NodeInstance node : this)
				if(node.level == i)
					string += "\n  " + node.node;
		}
		return string.trim();
	}
	
	/**
	 * An iterator for moving through the nodes in a subgraph.
	 * 
	 * @author Stephen G. Ware
	 */
	private final class MyIterator implements Iterator<NodeInstance> {

		/** The rest of the list not yet returned */
		Subgraph current = Subgraph.this;
		
		@Override
		public boolean hasNext() {
			return current.first != null;
		}

		@Override
		public NodeInstance next() {
			if(!hasNext())
				throw new NoSuchElementException("Iterator exhausted");
			NodeInstance element = current.first;
			current = current.rest;
			return element;
		}
	}

	@Override
	public Iterator<NodeInstance> iterator() {
		return new MyIterator();
	}
	
	/**
	 * Indicates how many nodes this subgraph contains.
	 * 
	 * @return the number of nodes
	 */
	public int size() {
		Subgraph current = this;
		int size = 0;
		while(current.first != null) {
			current = current.rest;
			size++;
		}
		return size;
	}
	
	/**
	 * Tests whether this subgraph contains a given node at the given level.
	 * Note that both node and level are checked, so if this method is called
	 * for literal A at level 2, and this subgraph contains A (but not at level
	 * 2), this method will return false.
	 * 
	 * @param node the node to search for
	 * @param level the level of the node being search for
	 * @return true if this subgraph contains this node at this level, false otherwise
	 */
	public boolean contains(Node node, int level) {
		Subgraph current = this;
		while(current.first != null) {
			if(current.first.node == node && current.first.level == level)
				return true;
			current = current.rest;
		}
		return false;
	}
	
	/**
	 * Returns a new subgraph which contains a given node at a given level.
	 * 
	 * @param node the node to add to the subgraph
	 * @param level the level at which to add the node to the subgraph
	 * @return a new subgraph containing that node at that level
	 */
	public Subgraph add(Node node, int level) {
		if(!node.exists(level))
			throw new IllegalArgumentException("The node \"" + node + "\" does not exist at level " + level + ".");
		else if(contains(node, level))
			return this;
		else
			return new Subgraph(new NodeInstance(node, level), this);
	}
	
	/**
	 * Returns a new subgraph which contains a given node at a given level.
	 * 
	 * @param instance specifies the node to add the level at which to add it
	 * @return a new subgraph containing that node at that level
	 */
	public Subgraph add(NodeInstance instance) {
		return add(instance.node, instance.level);
	}
	
	/**
	 * Returns a new subgraph which does not contain a given node at a given
	 * level.
	 * 
	 * @param node the node to be removed from the subgraph
	 * @param level the level at which to remove the node from the subgraph
	 * @return a new subgraph that does not contain that node at that level
	 */
	public Subgraph remove(Node node, int level) {
		return remove(this, node, level);
	}
	
	/**
	 * A private recursive helper method for removing an element from an
	 * immutable linked list.
	 * 
	 * @param current the current list
	 * @param node the node to be removed
	 * @param level the level at which the node should be removed
	 * @return a new subgraph with node removed, or the same subgraph if it did not contains that node at that level
	 */
	private static final Subgraph remove(Subgraph current, Node node, int level) {
		if(current.first == null)
			return current;
		else if(current.first.node == node && current.first.level == level)
			return current.rest;
		else {
			Subgraph rest = remove(current.rest, node, level);
			if(rest == current.rest)
				return current;
			else
				return new Subgraph(current.first, rest);
		}
	}
	
	/**
	 * Returns a new subgraph which does not contain a given node at a given
	 * level.
	 * 
	 * @param instance specifies the node to remove the level at which to remove it
	 * @return a new subgraph that does not contain that node at that level
	 */
	public Subgraph remove(NodeInstance instance) {
		return remove(instance.node, instance.level);
	}
	
	/**
	 * Converts the step nodes in this subgraph into a plan.
	 * 
	 * @return the plan this subgraph represents
	 */
	public Plan getPlan() {
		ArrayList<NodeInstance> nodes = new ArrayList<>();
		for(NodeInstance node : this)
			nodes.add(node);
		Collections.sort(nodes);
		ForwardPlan plan = new ForwardPlan();
		for(NodeInstance node : nodes) {
			if(node.node instanceof StepNode) {
				StepNode step = (StepNode) node.node;
				if(!step.persistence)
					plan = plan.addStep(((StepNode) node.node).step);
			}
		}
		return plan;
	}
}
