package edu.uky.ai.planning.pg;

import edu.uky.ai.SearchBudget;
import edu.uky.ai.planning.Plan;
import edu.uky.ai.planning.Problem;
import edu.uky.ai.planning.Search;

/**
 * Represents the two-phase search process of a {@link PlanGraphPlanner}. The
 * {@link #solve() solve} method performs the first phase, extending the plan
 * graph until all goals are non-mutex. It then calls
 * {@link PlanGraphPlanner#makeSearch(PlanGraph, edu.uky.ai.SearchBudget)} to
 * perform the second phase, searching the graph for a solution. If the second
 * phase does not return a solution, the graph is extended by one level and the
 * second phrase starts again with the extended plan graph. This continues
 * until a solution is found or the search runs out of resources.
 * 
 * @author Stephen G. Ware
 */
public class PlanGraphSearch extends Search {

	/** The planner which created this search object */
	private final PlanGraphPlanner planner;
	
	/** The plan graph to be extended and searched */
	private final PlanGraph graph;
	
	/** The current instance of the second phrase search object */
	private Search search = null;
	
	/** The total number of nodes visited across all phases */
	private int visited = 0;
	
	/** The total number of nodes expanded across all phases */
	private int generated = 0;
	
	/**
	 * Constructs a new plan graph search object for a given problem.
	 * 
	 * @param planner the planner which created this object
	 * @param problem the problem to be solved
	 */
	PlanGraphSearch(PlanGraphPlanner planner, Problem problem, SearchBudget budget) {
		super(problem, budget);
		this.planner = planner;
		this.graph = new PlanGraph(problem, true, true);
		this.graph.initialize(problem.initial);
	}

	@Override
	public int countVisited() {
		if(search == null)
			return visited;
		else
			return visited + search.countVisited();
	}

	@Override
	public int countGenerated() {
		if(search == null)
			return generated;
		else
			return generated + search.countGenerated();
	}

	@Override
	public final Plan solve() {
		while(!graph.goalAchieved() && !graph.hasLeveledOff())
			graph.extend();
		while(true) {
			if(budget.hasBeenExhausted()) {
				budget.checkTime();
				budget.incrementOperations();
			}
			if(search == null) {
				search = planner.makeSearch(graph, budget);
				if(search.budget != budget)
					throw new IllegalArgumentException("All subgraph searches must share a search budget.");
			}
			Plan plan = search.solve();
			if(plan == null) {
				visited += search.countVisited();
				generated += search.countGenerated();
				search = null;
				graph.extend();
			}
			else
				return plan;
		}
	}
}