package edu.uky.ai.planning.pg;

/**
 * Represents a plan graph node at a specific level.  {@link PlanGraph} saves
 * considerable memory by only using one object for each step and literal node,
 * rather than one object for every level at which those things appear.  This
 * class is used when we need to refer to a specific node at a specific level,
 * usually when searching for a solution subgraph.  For example, if the literal
 * node A has a level of 2, it indicates that A appears first at level 2, then
 * again at level 3, level 4, and all subsequence levels.  If we need to refer
 * specifically to the literal A that appears at level 3, we would use this
 * class.
 * 
 * @author Stephen G. Ware
 */
public class NodeInstance implements Comparable<NodeInstance> {

	/** The plan graph node */
	public final Node node;
	
	/** The level at which this node appears */
	public final int level;
	
	/**
	 * Constructs a new node instance for the given node and level.
	 * 
	 * @param node a node
	 * @param level a level in the plan graph
	 */
	public NodeInstance(Node node, int level) {
		this.node = node;
		this.level = level;
	}
	
	@Override
	public boolean equals(Object other) {
		if(other instanceof NodeInstance) {
			NodeInstance otherNI = (NodeInstance) other;
			return node == otherNI.node && level == otherNI.level;
		}
		return false;
	}
	
	@Override
	public int hashCode() {
		return node.hashCode() + level;
	}
	
	@Override
	public String toString() {
		return node + " at level " + level;
	}

	@Override
	public int compareTo(NodeInstance other) {
		return level - other.level;
	}
}
