package edu.uky.ai.planning.pg;

import java.util.ArrayList;
import java.util.Iterator;

import edu.uky.ai.logic.Literal;

/**
 * Represents a unique literal, or fact, in a plan graph.  Literal nodes have
 * producers ({@link StepNode step nodes} which have this literal as an effect)
 * and consumers ({@link StepNode step nodes} which have this literal as a
 * precondition).
 * 
 * @author Stephen G. Ware
 */
public class LiteralNode extends Node {

	/** The literal represented by this node */
	public final Literal literal;
	
	/** All step nodes which have this literal as an effect */
	protected final ArrayList<StepNode> producers = new ArrayList<>();
	
	/** All step nodes which have this literal as a precondition */
	protected final ArrayList<StepNode> consumers = new ArrayList<>();

	/**
	 * Constructs a new literal node for the given literal in the given plan
	 * graph.
	 * 
	 * @param graph the graph in which this node will exist
	 * @param literal the literal this node represents
	 */
	protected LiteralNode(PlanGraph graph, Literal literal) {
		super(graph);
		this.literal = literal;
	}
	
	@Override
	public int hashCode() {
		return literal.hashCode();
	}
	
	@Override
	public String toString() {
		return literal.toString();
	}
	
	@Override
	protected boolean setLevel(int level) {
		if(super.setLevel(level)) {
			for(StepNode consumer : consumers)
				consumer.incrementLiteralCount();
			return true;
		}
		else
			return false;
	}
	
	/**
	 * Returns all steps at a given level of the plan graph which have this
	 * literal as an effect.
	 * 
	 * @param level the index of a level in the plan graph
	 * @return all steps which exist at that level and which have this literal as an effect
	 * @throws IllegalArgumentException if this literal does not exist at the given level
	 */
	public Iterable<StepNode> getProducers(int level) {
		if(!exists(level))
			throw new IllegalArgumentException(this + " does not exist at level " + level + ".");
		return new Iterable<StepNode>() {
			@Override
			public Iterator<StepNode> iterator() {
				return new NodeIterator<>(level, producers);
			}
		};
	}
	
	/**
	 * Returns all steps at the next level of the plan graph which have this
	 * literal as a precondition.  Note that when this method is called for
	 * level n, it returns steps which exist at level n + 1.
	 * 
	 * @param level the index of a level in the plan graph at which this node exists
	 * @return all steps which exist at the next level and which have this literal as a precondition
	 * @throws IllegalArgumentException if this literal does not exist at the given level
	 */
	public Iterable<StepNode> getConsumers(int level) {
		if(!exists(level))
			throw new IllegalArgumentException(this + " does not exist at level " + level + ".");
		return new Iterable<StepNode>() {
			@Override
			public Iterator<StepNode> iterator() {
				return new NodeIterator<>(level + 1, consumers);
			}
		};
	}
}
