package edu.uky.ai.planning.io;

import java.util.ArrayList;
import java.util.LinkedHashMap;

import edu.uky.ai.io.FormatException;
import edu.uky.ai.io.List;
import edu.uky.ai.io.Node;
import edu.uky.ai.io.ObjectParser;
import edu.uky.ai.io.Parser;
import edu.uky.ai.io.Symbol;
import edu.uky.ai.logic.Constant;
import edu.uky.ai.logic.HashState;
import edu.uky.ai.logic.MutableState;
import edu.uky.ai.logic.Proposition;
import edu.uky.ai.logic.Term;
import edu.uky.ai.logic.Variable;
import edu.uky.ai.logic.io.PredicateLogicParser;
import edu.uky.ai.planning.Domain;
import edu.uky.ai.planning.Operator;
import edu.uky.ai.planning.Problem;

/**
 * A parser for planning {@link edu.uky.ai.planning.Domain}s and
 * {@link edu.uky.ai.planning.Problem}s in PDDL format.
 * 
 * @author Stephen G. Ware
 */
public class PlanningParser extends PredicateLogicParser {
	
	/** The keyword used to identify a domain */
	protected static final String DOMAIN_KEYWORD = ":domain";
	
	/** The keyword used to identify a domain's constants */
	protected static final String CONSTANTS_KEYWORD = ":constants";
	
	/** The keyword used to identify an operator */
	protected static final String OPERATOR_KEYWORD = ":action";
	
	/** The keyword used to identify an operator's parameters */
	protected static final String PARAMETERS_KEYWORD = ":parameters";
	
	/** The keyword used to identify an operator's precondition */
	protected static final String PRECONDITION_KEYWORD = ":precondition";
	
	/** The keyword used to identify an operator's effect */
	protected static final String EFFECT_KEYWORD = ":effect";
	
	/** The keyword used to identify a problem */
	protected static final String PROBLEM_KEYWORD = ":problem";
	
	/** The keyword used to identify a problem's objects */
	protected static final String OBJECTS_KEYWORD = ":objects";
	
	/** The keyword used to identify a problem's initial state */
	protected static final String INITIAL_STATE_KEYWORD = ":initial";
	
	/** The keyword used to identify a problem's goal */
	protected static final String GOAL_KEYWORD = ":goal";

	/**
	 * Constructs a new parser with rules for parsing planning objects.
	 */
	public PlanningParser() {
		setParser(Domain.class, DOMAIN_PARSER);
		setParser(Constant[].class, CONSTANTS_PARSER);
		setParser(Operator.class, OPERATOR_PARSER);
		setParser(Variable[].class, PARAMETERS_PARSER);
		setParser(Problem.class, PROBLEM_PARSER);
	}
	
	/** Parses domains */
	private static final ObjectParser<Domain> DOMAIN_PARSER = new ObjectParser<Domain>() {

		@Override
		public Domain parse(Node node, Parser parser) {
			if(!node.isList(2, -1) || !node.asList().first.isSymbol(DOMAIN_KEYWORD))
				return null;
			String name = node.asList().first.next.asSymbol().value;
			Constant[] constants = new Constant[0];
			Node constantsList = findList(CONSTANTS_KEYWORD, node.asList());
			if(constantsList != null)
				constants = parser.parse(constantsList, Constant[].class);
			for(Constant constant : constants)
				parser.setDefined(constant.name, constant);
			ArrayList<Operator> operators = new ArrayList<>();
			Node child = node.asList().first;
			while(child != null) {
				if(child.isList() && child.asList().first.isSymbol(OPERATOR_KEYWORD))
					operators.add(parser.parse(child, Operator.class));
				child = child.next;
			}
			return new Domain(name, constants, operators.toArray(new Operator[operators.size()]));
		}
	};
	
	/** Parses constants */
	private static final ObjectParser<Constant[]> CONSTANTS_PARSER = new ObjectParser<Constant[]>() {

		@Override
		public Constant[] parse(Node node, Parser parser) {
			if(!node.isList(2, -1) || !(node.asList().first.isSymbol(CONSTANTS_KEYWORD) || node.asList().first.isSymbol(OBJECTS_KEYWORD)))
				return null;
			ArrayList<Constant> constants = new ArrayList<>();
			node = node.asList().first.next;
			while(node != null) {
				if(node.asSymbol().value.startsWith("?"))
					throw new FormatException("Constant must not begin with \"?\"");
				String name = node.asSymbol().value;
				String type = Term.DEFAULT_TYPE;
				if(node.next != null && node.next.isSymbol("-")) {
					node = node.next.requireNext();
					type = node.asSymbol().value;
				}
				constants.add(new Constant(type, name));
				node = node.next;
			}
			return constants.toArray(new Constant[constants.size()]);
		}
	};
	
	/** Parses operators */
	private static final ObjectParser<Operator> OPERATOR_PARSER = new ObjectParser<Operator>() {

		@Override
		public Operator parse(Node node, Parser parser) {
			if(!node.isList(1, -1) || !node.asList().first.isSymbol(OPERATOR_KEYWORD))
				return null;
			String name = node.asList(8, 8).first.next.asSymbol().value;
			Variable[] parameters = parser.parse(requireValue(PARAMETERS_KEYWORD, node.asList().first), Variable[].class);
			for(Variable parameter : parameters)
				parser.setDefined(parameter.name, parameter);
			Proposition precondition = parser.parse(requireValue(PRECONDITION_KEYWORD, node.asList().first), Proposition.class);
			Proposition effect = parser.parse(requireValue(EFFECT_KEYWORD, node.asList().first), Proposition.class);
			return new Operator(name, parameters, precondition, effect);
		}
	};
	
	/** Parses a list of parameters */
	private static final ObjectParser<Variable[]> PARAMETERS_PARSER = new ObjectParser<Variable[]>() {

		@Override
		public Variable[] parse(Node node, Parser parser) {
			if(!node.isList())
				return null;
			ArrayList<Variable> parameters = new ArrayList<>();
			node = node.asList().first;
			while(node != null) {
				if(!node.asSymbol().value.startsWith("?"))
					throw new FormatException("Parameter must begin with \"?\"");
				String name = node.asSymbol().value.substring(1);
				String type = Term.DEFAULT_TYPE;
				if(node.next != null && node.next.isSymbol("-")) {
					node = node.next.requireNext();
					type = node.asSymbol().value;
				}
				parameters.add(new Variable(type, name));
				node = node.next;
			}
			return parameters.toArray(new Variable[parameters.size()]);
		}
	};
	
	/** Parses problems */
	private static final ObjectParser<Problem> PROBLEM_PARSER = new ObjectParser<Problem>() {

		@Override
		public Problem parse(Node node, Parser parser) {
			if(!node.isList(1, -1) || !node.asList().first.isSymbol(PROBLEM_KEYWORD))
				return null;
			String name = node.asList(3, -1).first.next.asSymbol().value;
			Domain domain = parser.requireDefined(node.asList().first.next.next.asList(2, 2).first.next.asSymbol().value, Domain.class);
			LinkedHashMap<String, Constant> objects = new LinkedHashMap<>();
			for(Constant constant : domain.constants)
				objects.put(constant.name, constant);
			List objectsList = findList(OBJECTS_KEYWORD, node.asList());
			if(objectsList != null)
				for(Constant constant : parser.parse(objectsList, Constant[].class))
					objects.put(constant.name, constant);
			for(Constant constant : objects.values())
				parser.setDefined(constant.name, constant);
			MutableState initial = new HashState();
			parser.parse(requireList(INITIAL_STATE_KEYWORD, node.asList()).first.requireNext(), Proposition.class).makeTrue(initial);
			Proposition goal = parser.parse(requireList(GOAL_KEYWORD, node.asList()).first.requireNext(), Proposition.class);
			return new Problem(name, domain, objects.values().toArray(new Constant[objects.size()]), initial, goal);
		}
	};	
	
	private static final List findList(String keyword, List list) {
		Node node = list.first;
		while(node != null) {
			if(node instanceof List && node.asList().first.isSymbol(keyword))
				return (List) node;
			node = node.next;
		}
		return null;
	}
	
	private static final List requireList(String keyword, List list) {
		List result = findList(keyword, list);
		if(result == null)
			throw new FormatException("Expected to find a list starting with \"" + keyword + "\" in \"" + list + "\".");
		else
			return result;
	}
	
	private static final Node findValue(String keyword, Node node) {
		while(node != null) {
			if(node.isSymbol(keyword))
				return node.requireNext();
			node = node.next;
		}
		return null;
	}
	
	private static final Node requireValue(String keyword, Node node) {
		Node result = findValue(keyword, node);
		if(result == null)
			throw new FormatException("Expected to find keyword \"" + keyword + "\" after \"" + (node instanceof Symbol ? ((Symbol) node).value : node) + "\".");
		else
			return result;
	}
}
