package edu.uky.ai.planning;

import java.util.Iterator;
import java.util.function.Consumer;

import edu.uky.ai.logic.Conjunction;
import edu.uky.ai.logic.Literal;
import edu.uky.ai.logic.Proposition;
import edu.uky.ai.util.ImmutableArray;

/**
 * A assortment of useful methods.
 * 
 * @author Stephen G. Ware
 */
public class Utilities {

	/**
	 * Since the precondition and effect of an action should be either a
	 * literal or a conjunction of literals, this utility method takes such
	 * a proposition and feeds each of its literals to a
	 * {@link java.util.function.Consumer}.
	 * 
	 * @param proposition the precondition or effect of an action
	 * @param consumer an operation that accepts a literal as input
	 */
	public static final void forEachLiteral(Proposition proposition, Consumer<? super Literal> consumer) {
		if(proposition instanceof Literal)
			consumer.accept((Literal) proposition);
		else if(proposition instanceof Conjunction)
			for(Proposition argument : ((Conjunction) proposition).arguments)
				consumer.accept((Literal) argument);
		else
			throw new UnsupportedOperationException("The proposition \"" + proposition + "\" is not a literal or a conjunction or literals.");
	}
	
	/**
	 * Since the precondition and effect of an action should be either a
	 * literal or a conjunction of literals, this utility method takes such
	 * a proposition and converts it into an {@link java.lang.Iterable}.
	 * 
	 * @param proposition the precondition or effect of an action
	 * @return an {@link java.lang.Iterable} of literals
	 */
	public static final Iterable<Literal> asLiterals(Proposition proposition) {
		return new LiteralIterable(proposition);
	}
	
	private static final class LiteralIterable implements Iterable<Literal> {

		private final Proposition proposition;
		
		LiteralIterable(Proposition proposition) {
			this.proposition = proposition;
		}
		
		@Override
		public Iterator<Literal> iterator() {
			if(proposition instanceof Literal)
				return new SingleLiteralIterator((Literal) proposition);
			else
				return new LiteralConjunctionIterator((Conjunction) proposition);
		}
	}
	
	private static final class SingleLiteralIterator implements Iterator<Literal> {

		private Literal literal;
		
		SingleLiteralIterator(Literal literal) {
			this.literal = literal;
		}
		
		@Override
		public boolean hasNext() {
			return literal != null;
		}

		@Override
		public Literal next() {
			Literal literal = this.literal;
			this.literal = null;
			return literal;
		}
	}
	
	private static final class LiteralConjunctionIterator implements Iterator<Literal> {

		private final ImmutableArray<Proposition> literals;
		private int index = 0;
		
		LiteralConjunctionIterator(Conjunction conjunction) {
			this.literals = conjunction.arguments;
		}
		
		@Override
		public boolean hasNext() {
			return index < literals.size();
		}

		@Override
		public Literal next() {
			return (Literal) literals.get(index++);
		}
	}
	
	/**
	 * Since the precondition and effect of an action should be either a
	 * literal or a conjunction of literals, this utility method takes such
	 * a proposition and converts it into an array of literals.
	 * 
	 * @param proposition the precondition or effect of an action
	 * @return an array of literals
	 */
	public static final Literal[] toLiterals(Proposition proposition) {
		if(proposition instanceof Literal)
			return new Literal[] { (Literal) proposition };
		else if(proposition instanceof Conjunction) {
			Conjunction conjunction = (Conjunction) proposition;
			Literal[] literals = new Literal[conjunction.arguments.size()];
			for(int i=0; i<conjunction.arguments.size(); i++)
				literals[i] = (Literal) conjunction.arguments.get(i);
			return literals;
		}
		else
			throw new UnsupportedOperationException("The proposition \"" + proposition + "\" is not a literal or a conjunction or literals.");
	}
}
