package edu.uky.ai.planning;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.function.Function;

import edu.uky.ai.SearchBudget;
import edu.uky.ai.io.DummyWriter;
import edu.uky.ai.planning.io.PlanningParser;
import edu.uky.ai.util.Arguments;
import edu.uky.ai.util.Table;
import edu.uky.ai.util.Utilities;
import edu.uky.ai.util.Table.Column;

/**
 * The entry point for the planning application.
 * 
 * @author Stephen G. Ware
 */
public class Main {
	
	private static final String USAGE = "Usage: java -jar planning.jar -a <planners> -d <domains> -p <problems> [-nl <nodes>] [-tl <millis>] [-o <file>]" +
		"\n  <planners> is one or more JAR files containing an instance of " + Planner.class.getName() +
		"\n  <domains>  is one or more planning domain files in PDDL format" +
		"\n  <problems> is one or more planning problem files in PDDL format" +
		"\n  <nodes>    is the optional maximum number of nodes each planner may visit during search" +
		"\n  <millis>   is the optional maximum number of milliseconds a planner may work on a problem" +
		"\n  <file>     is the file to which output will be written in HTML format";

	/**
	 * Launches the planning application according to its command line arguments.
	 * 
	 * @param args the command line arguments
	 * @throws Exception if any uncaught exceptions are thrown
	 */
	public static void main(String[] args) throws Exception {
		if(args.length < 6) {
			System.out.println(USAGE);
			System.exit(1);
		}
		Arguments arguments = new Arguments(args);
		ArrayList<Planner<?>> planners = new ArrayList<>();
		for(String url : arguments.getValues("-a"))
			planners.add(Utilities.loadFromJARFile(Planner.class, new File(url)));
		PlanningParser parser = new PlanningParser();
		for(String url : arguments.getValues("-d")) {
			System.out.println("Reading domain \"" + url + "\"...");
			Domain domain = parser.parse(new File(url), Domain.class);
			parser.setDefined(domain.name, domain);
		}
		ArrayList<Problem> problems = new ArrayList<>();
		for(String url : arguments.getValues("-p")) {
			System.out.println("Reading problem \"" + url + "\"...");
			problems.add(parser.parse(new File(url), Problem.class));
		}
		int maxNodes = SearchBudget.INFINITE_OPERATIONS;
		if(arguments.containsKey("-nl"))
			maxNodes = Integer.parseInt(arguments.getValue("-nl"));
		long maxTime = SearchBudget.INFINITE_TIME;
		if(arguments.containsKey("-tl"))
			maxTime = Long.parseLong(arguments.getValue("-tl"));
		final Writer output;
		if(arguments.containsKey("-o"))
			output = new BufferedWriter(new FileWriter(arguments.getValue("-o")));
		else
			output = new DummyWriter();
		try {
			if(planners.size() == 1 && problems.size() == 1)
				plan(planners.get(0), problems.get(0), maxNodes, maxTime, output);
			else
				benchmark(planners.toArray(new Planner[planners.size()]), problems.toArray(new Problem[problems.size()]), maxNodes, maxTime, output);
		}
		finally {
			output.close();
		}
	}
	
	/**
	 * Uses a given planner to solve a given problem.
	 * 
	 * @param planner the planner to use
	 * @param problem the problem to solve
	 * @param maxNodes the maximum nodes that may be visited during search
	 * @param maxTime the maximum milliseconds search may take
	 * @param output where the solution plan will be written (if one is found)
	 * @throws IOException if a problem occurs while writing to the output
	 */
	public static void plan(Planner<?> planner, Problem problem, int maxNodes, long maxTime, Writer output) throws IOException {
		Result result = planner.solve(problem, maxNodes, maxTime);
		System.out.println(
			"Planner:         " + result.planner.name + "\n" +
			"Domain:          " + result.problem.domain.name + "\n" +
			"Problem:         " + result.problem.name + "\n" +
			"Result:          " + result.reason + "\n" +
			"Nodes Visited:   " + result.visited + "\n" +
			"Nodes Generated: " + result.generated + "\n" +
			"Time (ms):       " + result.time
		);
		if(result.success) {
			System.out.println("\nSolution:");
			for(Step step : result.solution)
				System.out.println("  " + step);
		}
	}
	
	/**
	 * Compares the performance of one or more planners on one or more
	 * problems.
	 * 
	 * @param planners the planners to compare
	 * @param problems the problems to solve
	 * @param maxNodes the maximum nodes that a planner may visit when solving
	 * a problem
	 * @param maxTime the maximum milliseconds a planner may take when solving
	 * a problem
	 * @param output where output will be written in HTML format
	 * @throws IOException if a problem occurs while writing to the output
	 */
	public static void benchmark(Planner<?>[] planners, Problem[] problems, int maxNodes, long maxTime, Writer output) throws IOException {
		Table results = new Table(problems, planners);
		for(Problem problem : problems) {
			for(Planner<?> planner : planners) {
				System.out.print("Planner \"" + planner.name + "\" on problem \"" + problem.name + "\" in domain \"" + problem.domain.name + "\": ");
				Result result = planner.solve(problem, maxNodes, maxTime);
				results.getCell(problem, planner).value = result;
				System.out.println(result.reason);
			}
		}
		results = results.sortByColumn(BEST_PLANNERS);
		Table success = results.transform(SOLVED).addTotalRow().addTotalColumn();
		System.out.println("Results:\n" + success);
		output.append("<html>\n<head>\n<title>Planning Benchmark Results</title>");
		output.append("\n<style>\ntable { border-collapse: collapse; }\ntable, tr, th, td { border: 1px solid black; }\ntr:nth-child(odd) { background-color: lightgray; }\nth { font-weight: bold; }\ntd { text-align: right; }\n</style>");
		output.append("\n</head>\n<body>\n\n<h1>Planning Benchmark Results</h1>");
		output.append("\n\n<h2>Problems Solved</h2>\n" + success.toHTML());
		output.append("\n\n<h2>Plan Length</h2>\n" + results.transform(PLAN_SIZE).addTotalRow().addTotalColumn().toHTML());
		output.append("\n\n<h2>Nodes Visited</h2>\n" + results.transform(VISITED).addAverageRow().addAverageColumn().transform(TWO_DECIMAL_PLACES).toHTML());
		output.append("\n\n<h2>Nodes Generated</h2>\n" + results.transform(GENERATED).addAverageRow().addAverageColumn().transform(TWO_DECIMAL_PLACES).toHTML());
		output.append("\n\n<h2>Time (ms)</h2>\n" + results.transform(TIME).addAverageRow().addAverageColumn().transform(TWO_DECIMAL_PLACES).toHTML());
		output.append("\n\n</body>\n<html>");
		output.flush();
	}
	
	private static final Comparator<Table.Column> BEST_PLANNERS = new Comparator<Table.Column>() {
		@Override
		public int compare(Column column1, Column column2) {
			Number difference = Utilities.subtract(column2.sum(SOLVED), column1.sum(SOLVED));
			if(difference.doubleValue() == 0d)
				difference = Utilities.subtract(column1.sum(PLAN_SIZE), column2.sum(PLAN_SIZE));
			if(difference.doubleValue() == 0d)
				difference = Utilities.subtract(column1.sum(VISITED), column2.sum(VISITED));
			if(difference.doubleValue() == 0d)
				difference = Utilities.subtract(column1.sum(GENERATED), column2.sum(GENERATED));
			if(difference.doubleValue() == 0d)
				difference = Utilities.subtract(column1.sum(TIME), column2.sum(TIME));
			if(difference.doubleValue() == 0d)
				return 0;
			else if(difference.doubleValue() < 0d)
				return -1;
			else
				return 1;
		}
	};
	
	private static final Function<Object, Object> SOLVED = new Function<Object, Object>() {
		@Override
		public Object apply(Object object) {
			if(object instanceof Result)
				return ((Result) object).success ? 1 : 0;
			else if(object instanceof Problem)
				return Main.toString((Problem) object);
			else
				return object;
		}
	};
	
	private static final class ResultTransform implements Function<Object, Object> {
		
		private final Function<Result, Number> transform;
		
		public ResultTransform(Function<Result, Number> transform) {
			this.transform = transform;
		}
		
		@Override
		public Object apply(Object original) {
			if(original instanceof Result) {
				Result result = (Result) original;
				if(result.success)
					return transform.apply(result);
				else
					return "-";
			}
			else if(original instanceof Problem)
				return Main.toString((Problem) original);
			else
				return original;
		}
	}
	
	private static final Function<Object, Object> PLAN_SIZE = new ResultTransform(result -> result.solution.size());
	
	private static final Function<Object, Object> VISITED = new ResultTransform(result -> result.visited);
	
	private static final Function<Object, Object> GENERATED = new ResultTransform(result -> result.generated);
	
	private static final Function<Object, Object> TIME = new ResultTransform(result -> result.time);
	
	private static final Function<Object, String> TWO_DECIMAL_PLACES = new Function<Object, String>() {
		@Override
		public String apply(Object object) {
			if(object instanceof Number)
				return new DecimalFormat("0.00").format((Number) object);
			else
				return object.toString();
		}
	};
	
	private static final String toString(Problem problem) {
		return "(" + problem.domain.name + ") " + problem.name;
	}
}
