package edu.uky.ai.planning;

import java.util.Iterator;

import edu.uky.ai.util.ArrayIterator;

/**
 * A totally ordered plan which is built from beginning (first step) to end
 * last step). Note that this data structure is immutable.  The
 * {@link #addStep(Step)} method returns a new plan with the given step added
 * to the end without modifying the plan on which that method was called.
 * 
 * @author Stephen G. Ware
 */
public class ForwardPlan implements Plan {

	/** The rest of the steps in the plan */
	private final ForwardPlan first;
	
	/** The last step in the plan */
	private final Step last;
	
	/** The number of steps in the plan */
	private final int size;
	
	/**
	 * Constructs a new plan with a given rest of steps and last step.
	 * 
	 * @param first the rest of the plan
	 * @param last the last step
	 */
	private ForwardPlan(ForwardPlan first, Step last) {
		this.first = first;
		this.last = last;
		this.size = first.size + 1;
	}
	
	/**
	 * Constructs a new plan with 0 steps.
	 */
	public ForwardPlan() {
		this.first = null;
		this.last = null;
		this.size = 0;
	}
	
	@Override
	public String toString() {
		String str = "Plan:";
		for(Step step : this)
			str += "\n" + step;
		return str;
	}
	
	@Override
	public int size() {
		return size;
	}
	
	@Override
	public Iterator<Step> iterator() {
		Step[] steps = new Step[size];
		ForwardPlan current = this;
		for(int i = size - 1; i >= 0; i--) {
			steps[i] = current.last;
			current = current.first;
		}
		return new ArrayIterator<Step>(steps);
	}
	
	/**
	 * Returns a new plan with the given step added at the end.
	 * 
	 * @param step the next step to take
	 * @return a new plan whose last step is the given step
	 */
	public ForwardPlan addStep(Step step) {
		return new ForwardPlan(this, step);
	}
}
