package edu.uky.ai.planning;

import java.util.Iterator;

import edu.uky.ai.util.ArrayIterator;

/**
 * A totally ordered plan which is built from end (last step) to beginning.
 * In keeping with the contract of {@link Plan#iterator()}, this plan's
 * iterator still gives steps from first to last. Note that this data
 * structure is immutable.  The {@link #addStep(Step)} method returns a new
 * plan with the given step added to the beginning without modifying the plan
 * on which that method was called.
 * 
 * @author Stephen G. Ware
 */
public class BackwardPlan implements Plan {

	/** The first step in the plan */
	private final Step first;
	
	/** The rest of the steps in the plan */
	private final BackwardPlan rest;
	
	/** The number of steps in the plan */
	private final int size;
	
	/**
	 * Constructs a new plan with a given first step and rest of steps.
	 * 
	 * @param first the first step
	 * @param last the rest of the steps
	 */
	private BackwardPlan(Step first, BackwardPlan rest) {
		this.first = first;
		this.rest = rest;
		this.size = rest.size + 1;
	}
	
	/**
	 * Constructs a new plan with 0 steps.
	 */
	public BackwardPlan() {
		this.first = null;
		this.rest = null;
		this.size = 0;
	}
	
	@Override
	public String toString() {
		String str = "Plan:";
		for(Step step : this)
			str += "\n" + step;
		return str;
	}
	
	@Override
	public int size() {
		return size;
	}
	
	@Override
	public Iterator<Step> iterator() {
		Step[] steps = new Step[size];
		BackwardPlan current = this;
		for(int i=0; i<steps.length; i++) {
			steps[i] = current.first;
			current = current.rest;
		}
		return new ArrayIterator<Step>(steps);
	}
	
	/**
	 * Returns a new plan with the given step added at the beginning.
	 * 
	 * @param step the first step to take
	 * @return a new plan whose first step is the given step
	 */
	public BackwardPlan addStep(Step step) {
		return new BackwardPlan(step, this);
	}
}
