package edu.uky.ai.logic;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.List;

import edu.uky.ai.util.ImmutableArray;

final class Utilities {
	
	public static final int hashCode(String predicate, Iterable<?> arguments) {
		return predicate.hashCode() * 31 + arguments.hashCode();
	}
	
	public static final String toString(String predicate, Iterable<?> arguments) {
		String string = "(" + predicate;
		for(Object argument : arguments)
			string += " " + argument;
		return string + ")";
	}

	@SuppressWarnings("unchecked")
	public static final <F extends Formula> ImmutableArray<F> substitute(ImmutableArray<F> array, Substitution substitution) {
		for(int i=0; i<array.size(); i++) {
			F replacement = (F) array.get(i).substitute(substitution);
			if(array.get(i) != replacement) {
				F[] substituted = (F[]) Array.newInstance(array.getComponentType(), array.size());
				for(int j=0; j<i; j++)
					substituted[j] = (F) array.get(j);
				substituted[i] = replacement;
				for(int j=i+1; j<array.size(); j++)
					substituted[j] = (F) array.get(j).substitute(substitution);
				return new ImmutableArray<F>(substituted);
			}
		}
		return array;
	}
	
	public static final Proposition recombine(BooleanProposition proposition, NormalForm form) {
		Literal[][][] nf = new Literal[proposition.arguments.size()][][];
		for(int i=0; i<nf.length; i++)
			nf[i] = form.toArray(proposition.arguments.get(i));
		ArrayList<Literal[]> clauses = new ArrayList<>();
		cartesian(nf, 0, new Literal[nf.length][], clauses);
		return form.fromArray(clauses.toArray(new Literal[clauses.size()][]));
	}
	
	public static abstract class NormalForm {
		
		protected abstract Proposition convert(Proposition proposition);
		
		protected abstract boolean isClause(Proposition proposition);
		
		protected abstract Proposition makeClause(Literal[] literals);
		
		protected abstract Proposition makeProposition(Proposition[] arguments);
		
		protected Literal[][] toArray(Proposition proposition) {
			proposition = convert(proposition);
			if(proposition instanceof Literal || isClause(proposition))
				return new Literal[][] { clauseToArray(proposition) };
			else {
				BooleanProposition nf = (BooleanProposition) proposition;
				Literal[][] array = new Literal[nf.arguments.size()][];
				for(int i=0; i<array.length; i++)
					array[i] = clauseToArray(nf.arguments.get(i));
				return array;
			}
		}
		
		private Literal[] clauseToArray(Proposition proposition) {
			if(proposition instanceof Literal)
				return new Literal[] { (Literal) proposition };
			else {
				BooleanProposition clause = (BooleanProposition) proposition;
				Literal[] literals = new Literal[clause.arguments.size()];
				for(int i=0; i<literals.length; i++)
					literals[i] = (Literal) clause.arguments.get(i);
				return literals;
			}
		}
		
		protected Proposition fromArray(Literal[][] array) {
			if(array.length == 1)
				return arrayToClause(array[0]);
			else {
				Proposition[] arguments = new Proposition[array.length];
				for(int i=0; i<arguments.length; i++)
					arguments[i] = arrayToClause(array[i]);
				return makeProposition(arguments);
			}
		}
		
		private Proposition arrayToClause(Literal[] array) {
			if(array.length == 1)
				return array[0];
			else
				return makeClause(array);
		}
	}
	
	public static final NormalForm CNF = new NormalForm() {

		@Override
		protected Proposition convert(Proposition proposition) {
			return proposition.toCNF();
		}

		@Override
		protected boolean isClause(Proposition proposition) {
			return proposition instanceof Disjunction;
		}

		@Override
		protected Proposition makeClause(Literal[] literals) {
			return new Disjunction(literals);
		}

		@Override
		protected Proposition makeProposition(Proposition[] arguments) {
			return new Conjunction(arguments);
		}
	};
	
	public static final NormalForm DNF = new NormalForm() {

		@Override
		protected Proposition convert(Proposition proposition) {
			return proposition.toDNF();
		}

		@Override
		protected boolean isClause(Proposition proposition) {
			return proposition instanceof Conjunction;
		}

		@Override
		protected Proposition makeClause(Literal[] literals) {
			return new Conjunction(literals);
		}

		@Override
		protected Proposition makeProposition(Proposition[] arguments) {
			return new Disjunction(arguments);
		}
	};
	
	private static final void cartesian(Literal[][][] nf, int index, Literal[][] clauses, List<Literal[]> list) {
		if(index == nf.length) {
			ArrayList<Literal> literals = new ArrayList<>();
			for(Literal[] clause : clauses)
				for(Literal literal : clause)
					literals.add(literal);
			list.add(literals.toArray(new Literal[literals.size()]));
		}
		else {
			for(Literal[] clause : nf[index]) {
				clauses[index] = clause;
				cartesian(nf, index + 1, clauses, list);
			}
		}
	}
	
	/*
	public static final Proposition toNF(BooleanProposition proposition, Function<Proposition, Literal[][]> toArray) {
		Literal[][][] nf = new Literal[proposition.arguments.size()][][];
		for(int i=0; i<nf.length; i++)
			nf[i] = toArray.apply(proposition.arguments.get(i));
		ArrayList<Literal[]> clauses = new ArrayList<>();
		
		
	}
	
	private static final Function<Proposition, Literal[][]> CNFtoArray = new Function<Proposition, Literal[][]>() {
		@Override
		public Literal[][] apply(Proposition proposition) {
			if(proposition instanceof Literal || proposition instanceof Disjunction)
				return new Literal[][] { clauseToArray(proposition) };
			else {
				Conjunction conjunction = (Conjunction) proposition;
				Literal[][] cnf = new Literal[conjunction.arguments.size()][];
				for(int i=0; i<cnf.length; i++)
					cnf[i] = clauseToArray(conjunction.arguments.get(i));
				return cnf;
			}
		}
	};
	
	private static final Function<Proposition[], Conjunction> arrayToConjunction = new Function<Proposition[], Conjunction>() {
		@Override
		public Conjunction apply(Proposition[] arguments) {
			return new Conjunction(arguments);
		}
	};
	
	private static final Function<Proposition, Literal[][]> DNFtoArray = new Function<Proposition, Literal[][]>() {
		@Override
		public Literal[][] apply(Proposition proposition) {
			if(proposition instanceof Literal || proposition instanceof Conjunction)
				return new Literal[][] { clauseToArray(proposition) };
			else {
				Disjunction disjunction = (Disjunction) proposition;
				Literal[][] dnf = new Literal[disjunction.arguments.size()][];
				for(int i=0; i<dnf.length; i++)
					dnf[i] = clauseToArray(disjunction.arguments.get(i));
				return dnf;
			}
		}
	};
	
	private static final Function<Proposition[], Disjunction> arrayToDisjunction = new Function<Proposition[], Disjunction>() {
		@Override
		public Disjunction apply(Proposition[] arguments) {
			return new Disjunction(arguments);
		}
	};
	
	private static final Literal[] clauseToArray(Proposition proposition) {
		if(proposition instanceof Literal)
			return new Literal[] { (Literal) proposition };
		else {
			BooleanProposition b = (BooleanProposition) proposition;
			if(b.arguments.size() == 0) {
				if(b instanceof Conjunction)
					 return new Literal[] { Proposition.TRUE };
				else if(b instanceof Disjunction)
					return new Literal[] { Proposition.FALSE };
			}
			Literal[] clause = new Literal[b.arguments.size()];
			for(int i=0; i<clause.length; i++)
				clause[i] = (Literal) b.arguments.get(i);
			return clause;
		}
	}
	*/
}
