package edu.uky.ai.logic;

/**
 * The parent class of all logical formulas.
 * 
 * @author Stephen G. Ware
 */
public interface Formula extends Comparable<Formula> {

	@Override
	public default int compareTo(Formula other) {
		return toString().compareTo(other.toString());
	}
	
	/**
	 * Tests whether the formula represented by this object would be the same
	 * as another formula under the given substitution.
	 * 
	 * @param other the other logical formula to be compared to
	 * @param substitution a substitution
	 * @return true if the formulas would be the same after performing the
	 * substitution, false otherwise
	 */
	public default boolean equals(Formula other, Substitution substitution) {
		return this.substitute(substitution).equals(other.substitute(substitution));
	}
	
	/**
	 * Checks whether this formula contains any variables.
	 * 
	 * @return true if the formula is ground, false otherwise
	 */
	public boolean isGround();
	
	/**
	 * Returns a version of this formula such that any elements that appear in
	 * the given substitution are replaced.
	 * 
	 * @param substitution the substitution
	 * @return a formula with replacements from the substitution
	 */
	public Formula substitute(Substitution substitution);
	
	/**
	 * Finds the bindings that would be necessary to make this formula the
	 * same as another formula (if any exist).
	 * 
	 * @param other the other formula
	 * @param bindings an existing set of bindings that will be added to
	 * @return the bindings that would make both formulas the same, or null if
	 * so such bindings exist
	 */
	public Bindings unify(Formula other, Bindings bindings);
}
