package edu.uky.ai.logic;

import java.util.ArrayList;
import java.util.LinkedHashSet;

import edu.uky.ai.util.ImmutableArray;

/**
 * A disjunction, or logical OR statement, is true just when one or more of its
 * arguments are true.
 *
 * @author Stephen G. Ware
 */
public class Disjunction extends BooleanProposition {

	/** The word used to express a disjunction */
	public static final String DISJUNCTION_PREDICATE = "or";

	/**
	 * Constructs a disjunction from a group of arguments.
	 * 
	 * @param arguments the arguments
	 */
	public Disjunction(ImmutableArray<Proposition> arguments) {
		super(arguments);
	}

	/**
	 * Constructs a disjunction from a group of arguments.
	 * 
	 * @param arguments the arguments
	 */
	public Disjunction(Proposition...arguments) {
		this(new ImmutableArray<>(arguments));
	}
	
	/**
	 * Constructs a disjunction from a group of arguments.
	 * 
	 * @param arguments the arguments
	 */
	public Disjunction(Iterable<? extends Proposition> arguments) {
		this(edu.uky.ai.util.Utilities.toArray(arguments, Proposition.class));
	}
	
	@Override
	public boolean equals(Object other) {
		return other instanceof Disjunction && argumentsEqual(other);
	}
	
	@Override
	public int hashCode() {
		return Utilities.hashCode(DISJUNCTION_PREDICATE, arguments);
	}
	
	@Override
	public String toString() {
		return Utilities.toString(DISJUNCTION_PREDICATE, arguments);
	}
	
	@Override
	public Proposition substitute(Substitution substitution) {
		ImmutableArray<Proposition> arguments = substituteArguments(substitution);
		if(arguments == this.arguments)
			return (Proposition) substitution.get(this);
		else
			return (Proposition) substitution.get(new Disjunction(arguments));
	}
	
	@Override
	public Bindings unify(Formula other, Bindings bindings) {
		if(other instanceof Disjunction)
			return unifyArguments(other, bindings);
		else
			return null;
	}

	@Override
	public boolean isTrue(State state) {
		for(Proposition argument : arguments)
			if(argument.isTrue(state))
				return true;
		return false;
	}

	@Override
	public void makeTrue(MutableState state) {
		throw new IllegalStateException("Disjunctions are non-deterministic.");
	}

	@Override
	public Proposition simplify() {
		LinkedHashSet<Proposition> arguments = new LinkedHashSet<>();
		if(collect(this.arguments, arguments))
			return TRUE;
		else if(arguments.size() == 0)
			return FALSE;
		else if(arguments.size() == 1)
			return arguments.iterator().next();
		else
			return new Disjunction(arguments);
	}
	
	private static final boolean collect(ImmutableArray<Proposition> arguments, LinkedHashSet<Proposition> set) {
		for(Proposition argument : arguments) {
			argument = argument.simplify();
			if(argument == TRUE)
				return true;
			else if(argument == FALSE)
				continue;
			else if(argument instanceof Disjunction) {
				if(collect(((Disjunction) argument).arguments, set))
					return true;
			}
			else
				set.add(argument);
		}
		return false;
	}

	@Override
	public Disjunction negate() {
		return new Disjunction(negateArguments());
	}

	@Override
	public Proposition toCNF() {
		return Utilities.recombine(this, Utilities.CNF);
	}

	@Override
	public Proposition toDNF() {
		ArrayList<Proposition> clauses = new ArrayList<>();
		for(Proposition argument : arguments) {
			argument = argument.toDNF();
			if(argument instanceof Literal || argument instanceof Conjunction)
				clauses.add(argument);
			else
				for(Proposition clause : ((Disjunction) argument).arguments)
					clauses.add(clause);
		}
		return new Disjunction(clauses);
	}
}
