package edu.uky.ai.io;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;

import edu.uky.ai.util.ImmutableList;

/**
 * Converts text into various kinds of objects based on customizable rules.
 * 
 * @author Stephen G. Ware
 */
public class Parser implements Cloneable {
	
	private static final class Defined {
		
		public final String name;
		public final Object object;
		
		public Defined(String name, Object object) {
			this.name = name;
			this.object = object;
		}
	}

	private final HashMap<Class<?>, ObjectParser<?>> parsers;
	private ImmutableList<Defined> defined;
	
	/**
	 * Constructs a new parser that is a clone of the given parser.
	 * 
	 * @param toClone the parser to clone
	 */
	@SuppressWarnings("unchecked")
	protected Parser(Parser toClone) {
		this.parsers = (HashMap<Class<?>, ObjectParser<?>>) toClone.parsers.clone();
		this.defined = toClone.defined;
	}
	
	/**
	 * Constructs a new parser with no rules or defined objects.
	 */
	public Parser() {
		this.parsers = new HashMap<>();
		this.defined = new ImmutableList<>();
	}
	
	@Override
	public Parser clone() {
		return new Parser(this);
	}
	
	/**
	 * Returns the @{link ObjectParser} responsible for parsing objects of the
	 * given type.
	 * 
	 * @param <E> the type of object to be returned
	 * @param type the class for which a parser is needed
	 * @return the object parser for that class
	 */
	@SuppressWarnings("unchecked")
	public <E> ObjectParser<? extends E> getParser(Class<E> type) {
		ObjectParser<? extends E> parser = (ObjectParser<? extends E>) parsers.get(type);
		if(parser == null)
			throw new IllegalArgumentException("No method exists for parsing \"" + type.getName() + "\".");
		else
			return parser;
	}
	
	/**
	 * Sets the {@link ObjectParser} for a the given type.
	 * 
	 * @param <E> the type of object the object parser will handle
	 * @param type the class this object parser will handle
	 * @param parser the object parser
	 */
	public <E> void setParser(Class<E> type, ObjectParser<? extends E> parser) {
		parsers.put(type, parser);
	}
	
	/**
	 * Returns the most recently defined object whose names matches the given
	 * name and which is of a given type.
	 * 
	 * @param <E> the type of the desired object
	 * @param name the name of the desired object
	 * @param type the type of the desired object
	 * @return the defined object, or null if no such defined object exists
	 */
	@SuppressWarnings("unchecked")
	public <E> E getDefined(String name, Class<E> type) {
		ImmutableList<Defined> list = defined;
		while(list.size() != 0) {
			if(name.equals(list.first.name) && type.isAssignableFrom(list.first.object.getClass()))
				return (E) list.first.object;
			list = list.rest;
		}
		return null;
	}
	
	/**
	 * Like {@link #getDefined(String, Class)} but throws an exception if no
	 * object is found.
	 * 
	 * @param <E> the type of the desired object
	 * @param name the name of the desired object
	 * @param type the type of the desired object
	 * @return the defined object
	 * @throws FormatException if no defined object is found 
	 */
	public <E> E requireDefined(String name, Class<E> type) {
		E object = getDefined(name, type);
		if(object == null)
			throw new FormatException("No " + type.getSimpleName() + " named \"" + name + "\" defined.");
		else
			return object;
	}
	
	/**
	 * Defines an object by name.
	 * 
	 * @param name the name of the object being defined
	 * @param object the object (must be non-null)
	 */
	public void setDefined(String name, Object object) {
		if(object == null)
			throw new IllegalArgumentException("Cannot define null.");
		defined = defined.add(new Defined(name, object));
	}
	
	/**
	 * Parses a given file according to the rules defined by this parser and
	 * returns the object created.
	 * 
	 * @param <E> the type of object to be returned
	 * @param file the file to parse
	 * @param type the type of object this file should be parsed as
	 * @return the object that was created
	 * @throws IOException if a problem occurs when reading the file
	 * @throws FormatException if a problem occurs while parsing the file
	 */
	public <E> E parse(File file, Class<? extends E> type) throws IOException {
		return parse(Node.parse(file), type);
	}
	
	/**
	 * Parses a given string according to the rules defined by this parser and
	 * returns the object created.
	 * 
	 * @param <E> the type of object to be returned
	 * @param string the string to parse
	 * @param type the type of object this string should be parsed as
	 * @return the object that was created
	 * @throws FormatException if a problem occurs while parsing the string
	 */
	public <E> E parse(String string, Class<? extends E> type) {
		return parse(Node.parse(string), type);
	}
	
	/**
	 * Parses a given {@link Node} according to the rules defined by this parser
	 * and returns the object created.
	 * 
	 * @param <E> the type of object to be returned
	 * @param node the node to parse
	 * @param type the type of object this node should be parsed as
	 * @param types optional additional types to try to parse the node as if
	 * it cannot be parsed as the first type
	 * @return the object that was created
	 * @throws FormatException if a problem occurs while parsing the node
	 */
	@SafeVarargs
	public final <E> E parse(Node node, Class<? extends E> type, Class<? extends E>...types) {
		if(node == null)
			throw new FormatException("\"null\" cannot be parsed.");
		E object = getParser(type).parse(node, clone());
		if(object != null)
			return object;
		for(Class<? extends E> other : types) {
			object = getParser(other).parse(node, clone());
			if(object != null)
				return object;
		}
		String message = "\"" + node + "\" could not be parsed as " + type.getSimpleName();
		for(int i=0; i<types.length; i++) {
			if(types.length > 1)
				message += ",";
			message += " ";
			if(i == types.length - 1)
				message += "or ";
			message += types[i].getSimpleName();
		}
		message += ".";
		throw new FormatException(message);
	}
}
