 package edu.uky.ai.sat;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map.Entry;

import edu.uky.ai.logic.Atom;
import edu.uky.ai.logic.Conjunction;
import edu.uky.ai.logic.Disjunction;
import edu.uky.ai.logic.NegatedAtom;
import edu.uky.ai.logic.Proposition;
import edu.uky.ai.logic.io.PropositionalLogicParser;
import edu.uky.ai.util.ImmutableArray;
import edu.uky.ai.util.Utilities;

/**
 * Represents a logical formula which may or may not be satisfiable, organized
 * into clauses, literals, and variables.
 * 
 * The solution to a satisfiability problem is an {@link Assignment} where
 * enough variables have been set to true that the whole logical formula itself
 * is true.
 * 
 * This object represents the conjunction of {@link Clause}s that make up a
 * conjunctive normal form formula. It is true when all of its clauses are
 * true.
 * 
 * @author Stephen G. Ware
 */
public class Problem {

	/** The problem name */
	public final String name;
	
	/** The original logical formula, in conjunctive normal form and simplified */
	public final Conjunction proposition;
	
	/** All Boolean variables that appear somewhere in this problem */
	public final ImmutableArray<Variable> variables;
	
	/** The disjunctive clauses which make up the formula */
	public final ImmutableArray<Clause> clauses;
	
	final HashMap<Atom, Variable> variablesByAtom;
	private final int hashCode;
	
	/**
	 * Constructs a new problem with the given name for a given logical
	 * {@link edu.uky.ai.logic.Proposition}.
	 * 
	 * @param name the problem name
	 * @param proposition the logical formula
	 */
	public Problem(String name, Proposition proposition) {
		this.name = name;
		// Convert the proposition to conjunctive normal form.
		this.proposition = toCNF(proposition);
		// Find all unique atoms.
		LinkedHashMap<Atom, Integer> atoms = new LinkedHashMap<>();
		for(Proposition clause : this.proposition.arguments) {
			for(Proposition literal : ((Disjunction) clause).arguments) {
				Atom atom;
				if(literal == Proposition.TRUE || literal == Proposition.FALSE)
					continue;
				else if(literal instanceof Atom)
					atom = (Atom) literal;
				else
					atom = ((NegatedAtom) literal).argument;
				Integer count = atoms.get(atom);
				if(count == null)
					count = 0;
				atoms.put(atom, count + 1);
			}
		}
		// Make variable objects.
		HashMap<Atom, Variable> variablesByAtom = new HashMap<>();
		Variable[] variables = new Variable[atoms.size()];
		Literal[][] variableLiterals = new Literal[variables.length][];
		int index = 0;
		for(Entry<Atom, Integer> entry : atoms.entrySet()) {
			variableLiterals[index] = new Literal[entry.getValue()];
			variables[index] = new Variable(this, index, entry.getKey(), variableLiterals[index]);
			variablesByAtom.put(entry.getKey(), variables[index]);
			index++;
		}
		// Make clause objects.
		Clause[] clauses = new Clause[this.proposition.arguments.size()];
		Literal[][] clauseLiterals = new Literal[clauses.length][];
		index = 0;
		for(Atom atom : atoms.keySet())
			atoms.put(atom, 0);
		for(Proposition argument : this.proposition.arguments) {
			Disjunction clause = (Disjunction) argument;
			clauseLiterals[index] = new Literal[clause.arguments.size()];
			clauses[index] = new Clause(this, index, clause, clauseLiterals[index]);
			int j = 0;
			for(Proposition literal : clause.arguments) {
				if(literal instanceof NegatedAtom)
					clauseLiterals[index][j] = new Literal(clauses[index], (edu.uky.ai.logic.Literal) literal, false, variablesByAtom.get(((NegatedAtom) literal).argument));
				else
					clauseLiterals[index][j] = new Literal(clauses[index], (edu.uky.ai.logic.Literal) literal, true, variablesByAtom.get((Atom) literal));
				j++;
			}
			index++;
		}
		// Find each variable's literals.
		for(int i=0; i<variables.length; i++) {
			index = 0;
			for(Clause clause : clauses)
				for(Literal literal : clause.literals)
					if(variables[i] == literal.variable)
						variableLiterals[i][index++] = literal;
		}
		// Construct immutable arrays.
		this.variables = new ImmutableArray<>(variables);
		this.clauses = new ImmutableArray<>(clauses);
		this.variablesByAtom = variablesByAtom;
		this.hashCode = proposition.hashCode();
	}
	
	private static final Conjunction toCNF(Proposition proposition) {
		proposition = proposition.toCNF().simplify();
		if(proposition == Proposition.TRUE)
			return new Conjunction();
		else if(proposition == Proposition.FALSE)
			return new Conjunction(new Disjunction());
		else if(proposition instanceof edu.uky.ai.logic.Literal)
			proposition = new Disjunction(proposition);
		Conjunction conjunction;
		if(proposition instanceof Disjunction)
			conjunction = new Conjunction(proposition);
		else
			conjunction = (Conjunction) proposition;
		Disjunction[] clauses = new Disjunction[conjunction.arguments.size()];
		for(int i=0; i<clauses.length; i++) {
			if(conjunction.arguments.get(i) instanceof edu.uky.ai.logic.Literal)
				clauses[i] = new Disjunction(conjunction.arguments.get(i));
			else
				clauses[i] = (Disjunction) conjunction.arguments.get(i);
		}
		return new Conjunction(clauses);
	}
	
	/**
	 * Read a {@link edu.uky.ai.logic.Proposition} from a given file and
	 * constructs a problem from it.
	 * 
	 * @param file a file containing a logical proposition as text
	 * @throws IOException if a problem occurs while reading the file
	 */
	public Problem(File file) throws IOException {
		this(Utilities.getFileName(file), new PropositionalLogicParser().parse(file, Proposition.class));
	}
	
	@Override
	public int hashCode() {
		return hashCode;
	}
	
	@Override
	public String toString() {
		return name;
	}
	
	/**
	 * Returns a string representation of the proposition after substituting
	 * any variables whose values are known and simplifying the formula.
	 * 
	 * @param assignment an assignment that defines which variables have known
	 * values
	 * @return a string representation of the formula
	 */
	public String toString(Assignment assignment) {
		return proposition.substitute(assignment).simplify().toString();
	}
}
