package edu.uky.ai.sat;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.function.Function;

import edu.uky.ai.SearchBudget;
import edu.uky.ai.io.DummyWriter;
import edu.uky.ai.logic.Proposition;
import edu.uky.ai.util.Arguments;
import edu.uky.ai.util.Table;
import edu.uky.ai.util.Utilities;
import edu.uky.ai.util.Table.Column;

/**
 * The entry point for the satisfiability application.
 * 
 * @author Stephen G. Ware
 */
public class Main {

	private static final String USAGE = "Usage: java -jar sat.jar -a <solvers> -p <problems> [-ol <operations>] [-tl <millis>] [-o <file>]" +
		"\n  <solvers>    is one or more JAR files containing an instance of " + Solver.class.getName() +
		"\n  <problems>   is one or more files containing SAT expressions to solve" +
		"\n  <operations> is the optional maximum number of variable setting operations allowed" +
		"\n  <millis>     is the optional maximum number of milliseconds a solver may work on a problem" +
		"\n  <file>       is the file to which output will be written in HTML format";
	
	/**
	 * Launches the satisfiability program according to its command line arguments.
	 * 
	 * @param args the command line arguments
	 * @throws Exception if any uncaught exceptions are thrown
	 */
	public static void main(String[] args) throws Exception {
		if(args.length < 4) {
			System.out.println(USAGE);
			System.exit(1);
		}
		Arguments arguments = new Arguments(args);
		ArrayList<Solver> solvers = new ArrayList<>();
		for(String url : arguments.getValues("-a"))
			solvers.add(Utilities.loadFromJARFile(Solver.class, new File(url)));
		ArrayList<Problem> problems = new ArrayList<>();
		for(String url : arguments.getValues("-p")) {
			System.out.println("Reading file \"" + url + "\"...");
			problems.add(new Problem(new File(url)));
		}
		int maxOperations = SearchBudget.INFINITE_OPERATIONS;
		if(arguments.containsKey("-ol"))
			maxOperations = Integer.parseInt(arguments.getValue("-ol"));
		long maxTime = SearchBudget.INFINITE_TIME;
		if(arguments.containsKey("-tl"))
			maxTime = Long.parseLong(arguments.getValue("-tl"));
		final Writer output;
		if(arguments.containsKey("-o"))
			output = new BufferedWriter(new FileWriter(arguments.getValue("-o")));
		else
			output = new DummyWriter();
		try {
			if(solvers.size() == 1 && problems.size() == 1)
				solve(solvers.get(0), problems.get(0), maxOperations, maxTime, output);
			else
				benchmark(solvers.toArray(new Solver[solvers.size()]), problems.toArray(new Problem[problems.size()]), maxOperations, maxTime, output);
		}
		finally {
			output.close();
		}
	}
	
	/**
	 * Runs a single satisfiability solver on a single problem.
	 * 
	 * @param solver the solver to run
	 * @param problem the problem to solve
	 * @param maxOperations the maximum number of times that variable values
	 * can be set during search
	 * @param maxTime the maximum milliseconds search may take
	 * @param output where the satisfying assignment will be written (if one
	 * is found)
	 * @throws IOException if a problem occurs while writing to the output
	 */
	public static void solve(Solver solver, Problem problem, int maxOperations, long maxTime, Writer output) throws IOException {
		Result result = solver.solve(problem, maxOperations, maxTime);
		System.out.println(
			"Solver:     " + result.solver.name + "\n" +
			"Problem:    " + result.problem.name + "\n" +
			"Result:     " + result.reason + "\n" +
			"Operations: " + result.operations + "\n" +
			"Time (ms):  " + result.time
		);
		if(result.success) {
			Proposition solution = result.solution.toProposition();
			System.out.println("Solution:   " + solution);
			output.append(solution.toString());
		}
	}
	
	/**
	 * Compares the performance of one or more solvers on one or more problems.
	 * 
	 * @param solvers the satisfiability solvers to compare
	 * @param problems the problems to solve
	 * @param maxOperations the maximum number of times that variable values
	 * can be set during search
	 * @param maxTime the maximum milliseconds a planner may take when solving
	 * a problem
	 * @param output where output will be written in HTML format
	 * @throws IOException if a problem occurs while writing to the output
	 */
	public static void benchmark(Solver[] solvers, Problem[] problems, int maxOperations, long maxTime, Writer output) throws IOException {
		Table results = new Table(problems, solvers);
		for(Problem problem : problems) {
			for(Solver solver : solvers) {
				System.out.print("Solver \"" + solver.name + "\" on problem \"" + problem.name + "\": ");
				Result result = solver.solve(problem, maxOperations, maxTime);
				results.getCell(problem, solver).value = result;
				System.out.println(result.reason);
			}
		}
		results = results.sortByColumn(BEST_SOLVERS);
		Table success = results.transform(SOLVED).addTotalRow().addTotalColumn();
		System.out.println("Results:\n" + success);
		output.append("<html>\n<head>\n<title>Satisfiability Benchmark Results</title>");
		output.append("\n<style>\ntable { border-collapse: collapse; }\ntable, tr, th, td { border: 1px solid black; }\ntr:nth-child(odd) { background-color: lightgray; }\nth { font-weight: bold; }\ntd { text-align: right; }\n</style>");
		output.append("\n</head>\n<body>\n\n<h1>Satisfiability Benchmark Results</h1>");
		output.append("\n\n<h2>Problems Solved</h2>\n" + success.toHTML());
		output.append("\n\n<h2>Solution Size</h2>\n" + results.transform(SIZE).addTotalRow().addTotalColumn().toHTML());
		output.append("\n\n<h2>Operations</h2>\n" + results.transform(OPERATIONS).addAverageRow().addAverageColumn().transform(TWO_DECIMAL_PLACES).toHTML());
		output.append("\n\n<h2>Time (ms)</h2>\n" + results.transform(TIME).addAverageRow().addAverageColumn().transform(TWO_DECIMAL_PLACES).toHTML());
		output.append("\n\n</body>\n<html>");
		output.flush();
	}
	
	private static final Comparator<Table.Column> BEST_SOLVERS = new Comparator<Table.Column>() {
		@Override
		public int compare(Column column1, Column column2) {
			Number difference = Utilities.subtract(column2.sum(SOLVED), column1.sum(SOLVED));
			if(difference.doubleValue() == 0d)
				difference = Utilities.subtract(column1.sum(SIZE), column2.sum(SIZE));
			if(difference.doubleValue() == 0d)
				difference = Utilities.subtract(column1.sum(OPERATIONS), column2.sum(OPERATIONS));
			if(difference.doubleValue() == 0d)
				difference = Utilities.subtract(column1.sum(TIME), column2.sum(TIME));
			if(difference.doubleValue() == 0d)
				return 0;
			else if(difference.doubleValue() < 0d)
				return -1;
			else
				return 1;
		}
	};	
	
	private static final Function<Object, Object> SOLVED = new Function<Object, Object>() {
		@Override
		public Object apply(Object object) {
			if(object instanceof Result)
				return ((Result) object).success ? 1 : 0;
			else
				return object;
		}
	};
	
	private static final class ResultTransform implements Function<Object, Object> {
		
		private final Function<Result, Number> transform;
		
		public ResultTransform(Function<Result, Number> transform) {
			this.transform = transform;
		}
		
		@Override
		public Object apply(Object original) {
			if(original instanceof Result) {
				if(((Result) original).success)
					return transform.apply((Result) original);
				else
					return "-";
			}
			else
				return original;
		}
	}
	
	private static final ResultTransform SIZE = new ResultTransform(result -> {
		int total = 0;
		for(Variable variable : result.problem.variables)
			if(result.solution.getValue(variable) == Value.TRUE)
				total++;
		return total;
	});
	
	private static final ResultTransform OPERATIONS = new ResultTransform(result -> result.operations);
	
	private static final ResultTransform TIME = new ResultTransform(result -> result.time);
	
	private static final Function<Object, String> TWO_DECIMAL_PLACES = new Function<Object, String>() {
		@Override
		public String apply(Object object) {
			if(object instanceof Number)
				return new DecimalFormat("0.00").format((Number) object);
			else
				return object.toString();
		}
	};
}
