package edu.uky.ai.logic;

/**
 * A term is a reference to an individual thing in the world, and does not have
 * a truth value. Terms have types, which define groups of similar things. Two
 * terms are considered to refer to the same thing (i.e. are
 * {@link #equals(Object)}) if they have the same type and same name.
 * 
 * @author Stephen G. Ware
 */
public abstract class Term implements Formula {
	
	/** The default type. All terms are considered to be of this type. */
	public static final String DEFAULT_TYPE = "object";
	
	/**
	 * The term's type. All terms are of the {@link #DEFAULT_TYPE} and may
	 * also be of one additional type as defined by this string.
	 */
	public final String type;
	
	/** The thing's name */
	public final String name;
	
	/**
	 * Constructs a term from the given type and name.
	 * 
	 * @param type the term's type
	 * @param name the term's name
	 */
	public Term(String type, String name) {
		this.type = type;
		this.name = name;
	}
	
	@Override
	public boolean equals(Object other) {
		if(getClass() == other.getClass()) {
			Term otherTerm = (Term) other;
			return type.equals(otherTerm.type) && name.equals(otherTerm.name);
		}
		return false;
	}
	
	@Override
	public int hashCode() {
		return type.hashCode() + name.hashCode();
	}
	
	@Override
	public String toString() {
		return name;
	}

	@Override
	public Term substitute(Substitution substitution) {
		return (Term) substitution.get(this);
	}
	
	@Override
	public Bindings unify(Formula other, Bindings bindings) {
		if(other instanceof Term)
			return bindings.setEqual(this, (Term) other);
		else
			return null;
	}
}
