package edu.uky.ai.logic;

/**
 * A state fully describes the disposition of all things in a world. For every
 * {@link Atom} that is meaningful in that world, a state defines whether that
 * atom is true or false.
 * 
 * @author Stephen G. Ware
 */
public interface State extends Cloneable, Iterable<Atom> {

	/**
	 * Check whether or not an atom is true in this state.
	 * 
	 * @param atom the atom whose truth value will be checked
	 * @return true if the atom is true in this state, false otherwise
	 */
	public boolean contains(Atom atom);
	
	/**
	 * Represents this state as a proposition which would be true in this state
	 * and false in any other state (typically a {@link Conjunction} of all
	 * {@link Atom}s that are currently true). This proposition uses the Closed
	 * World Assumption, meaning that atoms not explicitly stated to be true
	 * can be assumed false.
	 * 
	 * @return a proposition representing this state
	 */
	public default Proposition toProposition() {
		return new Conjunction(this);
	}
}
