package edu.uky.ai.logic;

import edu.uky.ai.util.ImmutableArray;

/**
 * The atoms of predicate logic, which are composed of statements that express
 * relationships between things in the world (i.e. {@link Term}s).
 * 
 * @author Stephen G. Ware
 */
public class Predication implements Atom {

	/** The relationship between the terms, (often a preposition) */
	public final String predicate;
	
	/** The terms between which the relationship holds */
	public final ImmutableArray<Term> terms;

	/**
	 * Constructs a new predication with the given predicate and terms.
	 * 
	 * @param predicate the relationship between the terms
	 * @param terms the terms between which the relationship holds
	 */
	public Predication(String predicate, ImmutableArray<Term> terms) {
		this.predicate = predicate;
		this.terms = terms;
	}
	
	/**
	 * Constructs a new predication with the given predicate and terms.
	 * 
	 * @param predicate the relationship between the terms
	 * @param terms the terms between which the relationship holds
	 */
	public Predication(String predicate, Term...terms) {
		this(predicate, new ImmutableArray<>(terms));
	}
	
	/**
	 * Constructs a new predication with the given predicate and terms.
	 * 
	 * @param predicate the relationship between the terms
	 * @param terms the terms between which the relationship holds
	 */
	public Predication(String predicate, Iterable<Term> terms) {
		this(predicate, edu.uky.ai.util.Utilities.toArray(terms, Term.class));
	}
	
	@Override
	public boolean equals(Object other) {
		if(other instanceof Predication) {
			Predication otherPredication = (Predication) other;
			if(predicate.equals(otherPredication.predicate) && terms.size() == otherPredication.terms.size()) {
				for(int i=0; i<terms.size(); i++)
					if(!terms.get(i).equals(otherPredication.terms.get(i)))
						return false;
				return true;
			}
		}
		return false;
	}
	
	@Override
	public int hashCode() {
		return Utilities.hashCode(predicate, terms);
	}
	
	@Override
	public String toString() {
		return Utilities.toString(predicate, terms);
	}
	
	@Override
	public boolean isGround() {
		for(Term term : terms)
			if(!term.isGround())
				return false;
		return true;
	}
	
	@Override
	public Atom substitute(Substitution substitution) {
		ImmutableArray<Term> terms = Utilities.substitute(this.terms, substitution);
		if(terms == this.terms)
			return (Atom) substitution.get(this);
		else
			return (Atom) substitution.get(new Predication(predicate, terms));
	}

	@Override
	public Bindings unify(Formula other, Bindings bindings) {
		if(!(other instanceof Predication))
			return null;
		Predication otherPredication = (Predication) other;
		if(!predicate.equals(otherPredication.predicate) || terms.size() != otherPredication.terms.size())
			return null;
		for(int i=0; i<terms.size() && bindings != null; i++)
			bindings = terms.get(i).unify(otherPredication.terms.get(i), bindings);
		return bindings;
	}
}
