package edu.uky.ai.logic;

import edu.uky.ai.util.ImmutableArray;

/**
 * Represents the negation (or opposite) of a proposition. A negation is true
 * just when it argument is false, and false just when its argument is true.
 * 
 * @author Stephen G. Ware
 */
public class Negation extends BooleanProposition {
	
	/** The word used to express a negation */
	public static final String NEGATION_PREDICATE = "not";

	/** The proposition this formula is the opposite of */
	public final Proposition argument;
	
	/**
	 * Constructs a new negation with the given proposition.
	 * 
	 * @param argument the proposition this formula will be the opposite of
	 */
	public Negation(Proposition argument) {
		super(new ImmutableArray<>(new Proposition[] { argument }));
		this.argument = argument;
	}
	
	@Override
	public boolean equals(Object other) {
		return other instanceof Negation && argumentsEqual(other);
	}
	
	@Override
	public int hashCode() {
		return Utilities.hashCode(NEGATION_PREDICATE, arguments);
	}
	
	@Override
	public String toString() {
		return Utilities.toString(NEGATION_PREDICATE, arguments);
	}

	@Override
	public Proposition substitute(Substitution substitution) {
		ImmutableArray<Proposition> arguments = substituteArguments(substitution);
		if(arguments == this.arguments)
			return (Proposition) substitution.get(this);
		else
			return (Proposition) substitution.get(new Negation(arguments.get(0)));
	}
	
	@Override
	public Bindings unify(Formula other, Bindings bindings) {
		if(other instanceof Negation)
			return unifyArguments(other, bindings);
		else
			return null;
	}

	@Override
	public boolean isTrue(State state) {
		return !argument.isTrue(state);
	}

	@Override
	public void makeTrue(MutableState state) {
		simplify().makeTrue(state);
	}

	@Override
	public Proposition simplify() {
		return argument.negate();
	}

	@Override
	public Proposition negate() {
		return argument;
	}

	@Override
	public Proposition toCNF() {
		return simplify().toCNF();
	}

	@Override
	public Proposition toDNF() {
		return simplify().toDNF();
	}
}
