package edu.uky.ai.logic;

import java.util.HashSet;

import edu.uky.ai.util.ImmutableList;

/**
 * An implementation of {@link Bindings} based on a linked list. This data
 * structure is immutable, meaning that operations which would modify it return
 * a new object and leave the original unchanged.
 * 
 * @author Stephen G. Ware
 */
public class ListBindings implements Bindings {

	/**
	 * An entry stores three things: a single constant called the 'value' (or
	 * null if no value), a list of variables that are all equal to one
	 * another and to the value, and a list of terms (variables or constants)
	 * that are not equal to any of the variables in the first list or to the
	 * value.
	 * 
	 * @author Stephen G. Ware
	 */
	private static final class Entry {
		
		/** An entry with no value, no equal variables, and no not equal variables */
		public static final Entry EMPTY = new Entry(null, Term.DEFAULT_TYPE, new ImmutableList<>(), new ImmutableList<>());
		
		/** The constant value of this entry, or null if no value */
		public final Constant value;
		
		/** The object type of this entry */
		public final String type;
		
		/** A list of variables all equal to one another and to the value (if any) */
		public final ImmutableList<Variable> equalSet;
		
		/** A list of terms all not equal to the variables in the equal set and not equal to the value (if any) */
		public final ImmutableList<Term> notEqualSet;
		
		/**
		 * Creates an entry with the given value, equal set, and not equal set.
		 * 
		 * @param value the value (or null if no value)
		 * @param equalSet the equal variables
		 * @param notEqualSet the not equal terms
		 */
		private Entry(Constant value, String type, ImmutableList<Variable> equalSet, ImmutableList<Term> notEqualSet) {
			this.value = value;
			this.type = type;
			this.equalSet = equalSet;
			this.notEqualSet = notEqualSet;
		}
		
		@Override
		public String toString() {
			String str = "";
			boolean first = true;
			for(Variable variable : equalSet) {
				if(first)
					first = false;
				else
					str += " == ";
				str += variable;
			}
			if(value != null)
				str += " == " + value;
			for(Term term : notEqualSet)
				str += " != " + term;
			return str;
		}
		
		/**
		 * Returns a new entry with the value set to a given constant.
		 * 
		 * @param value the value the new entry will have
		 * @return an identical entry, but with the given value
		 */
		public Entry setValue(Constant value) {
			String type = this.type.equals(Term.DEFAULT_TYPE) ? value.type : this.type;
			return new Entry(value, type, this.equalSet, this.notEqualSet);
		}
		
		/**
		 * Returns a new entry with a given variable added to the equal set.
		 * 
		 * @param variable the variable to be added to the equal set
		 * @return an identical entry, but with the variable in the equal set
		 */
		public Entry addEqual(Variable variable) {
			String type = this.type.equals(Term.DEFAULT_TYPE) ? variable.type : this.type;
			return new Entry(value, type, equalSet.add(variable), notEqualSet);
		}
		
		/**
		 * Returns a new entry with a given term added to the not equal set.
		 * 
		 * @param term the term to be added to the not equal set
		 * @return an identical entry, but with the term in the not equal set
		 */
		public Entry addNotEqual(Term term) { 
			return new Entry(value, type, equalSet, notEqualSet.add(term));
		}
		
		/**
		 * Merged two entries together, if possible.
		 * 
		 * @param other the other entry to merge with this one
		 * @return the merged entry, or null if the merger is impossible
		 */
		public Entry merge(Entry other) {
			// If both entries have been set to different values, the merger is
			// impossible.
			if(value != null && other.value != null && !value.equals(other.value))
				return null;
			Constant value = this.value == null ? other.value : this.value;
			String type = this.type.equals(Term.DEFAULT_TYPE) ? other.type : this.type;
			// Combine the equal sets of both entries.
			ImmutableList<Variable> equalSet = this.equalSet;
			for(Variable eq : other.equalSet)
				if(!equalSet.contains(eq))
					equalSet = equalSet.add(eq);
			// Check types.
			if(value != null && !is(value.type, type))
				return null;
			for(Variable eq : equalSet)
				if(!is(type, eq.type))
					return null;
			// Combine the not equal sets of both entries.
			ImmutableList<Term> notEqualSet = this.notEqualSet;
			for(Term neq : other.notEqualSet)
				if(!notEqualSet.contains(neq))
					notEqualSet = notEqualSet.add(neq);
			// If any term in the combined not equal set is the value or
			// appears in the equal set, the merger is impossible.
			for(Term neq : notEqualSet)
				if((value != null && neq.equals(value)) || equalSet.contains(neq))
					return null;
			return new Entry(value, type, equalSet, notEqualSet);
		}
		
		/**
		 * Check if the first type is the same as or more specific than the
		 * second type.
		 * 
		 * @param t1 the first type
		 * @param t2 the second type
		 * @return true if t1 is the same as or a subtype of t2, false otherwise
		 */
		private static final boolean is(String t1, String t2) {
			if(t2.equals(Term.DEFAULT_TYPE))
				return true;
			else
				return t1.equals(t2);
		}
	}
	
	/** An empty set of bindings */
	public static final ListBindings EMPTY = new ListBindings(null, null);
	
	/** The first entry in the list */
	private final Entry first;
	
	/** The rest of the linked list */
	private final ListBindings rest;
	
	/**
	 * Constructs a new ListBindings with a given first entry and a given rest
	 * of the list.
	 * 
	 * @param first the first entry
	 * @param rest the rest of the linked list
	 */
	private ListBindings(Entry first, ListBindings rest) {
		this.first = first;
		this.rest = rest;
	}
	
	@Override
	public String toString() {
		String str = "";
		HashSet<Variable> variables = new HashSet<>();
		ListBindings current = this;
		boolean first = true;
		while(current.first != null) {
			boolean skip = true;
			for(Variable variable : current.first.equalSet)
				if(!variables.contains(variable))
					skip = false;
			if(!skip) {
				if(first)
					first = false;
				else
					str += "; ";
				str += current.first;
				for(Variable variable : current.first.equalSet)
					variables.add(variable);
			}
			current = current.rest;
		}
		return str;
	}
	
	/**
	 * Searches the list for the entry associated with a given term.  If no
	 * entry can be found, a new one is created.
	 * 
	 * @param term the term to find
	 * @return the entry associated with that term
	 */
	private final Entry find(Term term) {
		ListBindings current = this;
		while(current.first != null) {
			if((current.first.value != null && current.first.value.equals(term)) || current.first.equalSet.contains(term))
				return current.first;
			current = current.rest;
		}
		if(term instanceof Constant)
			return Entry.EMPTY.setValue((Constant) term);
		else
			return Entry.EMPTY.addEqual((Variable) term);
	}

	@Override
	public Formula get(Formula original) {
		if(original instanceof Term && !(original instanceof Constant)) {
			Entry group = find((Term) original);
			if(group.value == null)
				return original;
			else
				return group.value;
		}
		return original;
	}

	@Override
	public Bindings setEqual(Term t1, Term t2) {
		Entry merged = find(t1).merge(find(t2));
		if(merged == null)
			return null;
		else
			return new ListBindings(merged, this);
	}

	@Override
	public Bindings setNotEqual(Term t1, Term t2) {
		Entry entry1 = find(t1).addNotEqual(t2);
		entry1 = entry1.merge(entry1);
		if(entry1 == null)
			return null;
		Entry entry2 = find(t2).addNotEqual(t1);
		entry2 = entry2.merge(entry2);
		if(entry2 == null)
			return null;
		return new ListBindings(entry1, new ListBindings(entry2, this));
	}
}