package edu.uky.ai.util;

import java.util.Comparator;
import java.util.PriorityQueue;

/**
 * A min priority queue which, instead of using a {@link Comparator}, allows
 * the user to specify a numeric key that should be associated with the item
 * being added to the queue.  When an item is popped off this queue, it will
 * always be the item with the lowest key.  This queue may hold multiple copies
 * of the same object with different keys.  This class relies on
 * {@link java.util.PriorityQueue}.
 * 
 * @author Stephen G. Ware
 * @param <T> the type of object this queue will hold
 */
public class MinPriorityQueue<T> {

	/** A counter for assigning a unique ID number to each node */
	private static int nextID = 0;
	
	/**
	 * A node holds an object and its key.
	 * 
	 * @author Stephen G. Ware
	 */
	private final class Node {
		
		/** A unique ID number for this node, where lower IDs were created earlier */
		public final int id = nextID++;
		
		/** The object associated with this node */
		public final T element;
		
		/** The object's key */
		public final double priority;
		
		/**
		 * Constructs a new node with the given object and key.
		 * 
		 * @param element the object
		 * @param priority the key
		 */
		Node(T element, double priority) {
			this.element = element;
			this.priority = priority;
		}
	}
	
	/** The queue in which nodes are stored */
	private final PriorityQueue<Node> queue;
	
	/**
	 * Constructs a new, empty min priority queue.
	 */
	public MinPriorityQueue() {
		queue = new PriorityQueue<Node>(new Comparator<Node>(){
			@Override
			public int compare(MinPriorityQueue<T>.Node n1, MinPriorityQueue<T>.Node n2) {
				double comparison = n1.priority - n2.priority;
				if(comparison == 0)
					comparison = n1.id - n2.id;
				if(comparison < 0)
					return -1;
				else if(comparison > 0)
					return 1;
				else
					return 0;
			}
		});
	}
	
	/**
	 * Tests whether this queue is empty.
	 * 
	 * @return true if it is empty, false otherwise
	 */
	public boolean isEmpty() {
		return queue.isEmpty();
	}
	
	/**
	 * Returns the number of objects currently stored in this queue.
	 * 
	 * @return the size
	 */
	public int size() {
		return queue.size();
	}
	
	/**
	 * Returns, but does not remove, the object with the lowest key value (i.e.
	 * the next object to be removed).
	 * 
	 * @return the object
	 */
	public T peek() {
		return queue.peek().element;
	}
	
	/**
	 * Associates an object with a given key and adds it to the queue.
	 * 
	 * @param element the object to add
	 * @param priority the key to associate with the object
	 */
	public void push(T element, double priority) {
		queue.add(new Node(element, priority));
	}
	
	/**
	 * Removes and returns to the object with the lowest key.
	 * 
	 * @return the object
	 */
	public T pop() {
		return queue.poll().element;
	}
}
