package edu.uky.ai.sl;

import edu.uky.ai.data.LabeledDataSet;
import edu.uky.ai.util.Utilities;

/**
 * Summarizes the overall performance of a {@link Learner} on a
 * {@link edu.uky.ai.data.LabeledDataSet labeled data set} after a
 * comprehensive evaluation, such as 10-fold cross-validation
 * (i.e. a summary of many {@link Result}s).
 * 
 * @author Stephen G. Ware
 */
public class Summary {

	/** The learner */
	public final Learner learner;
	
	/** The data set used to train and test the model */
	public final LabeledDataSet data;
	
	/** The most accurate model learned across all folds */
	public final Model model;
	
	/** The total number of test points correctly classified across all folds */
	public final int correct;
	
	/** The average accuracy across all folds */
	public final double accuracy;
	
	/** The total time in milliseconds taken for all folds */
	public final long time;
	
	/**
	 * Constructs a summary from many {@link Result}s.
	 * 
	 * @param learner the learner
	 * @param data the data set
	 * @param results the results of each individual fold
	 */
	public Summary(Learner learner, LabeledDataSet data, Result...results) {
		this.learner = learner;
		this.data = data;
		Model model = null;
		int correct = 0;
		double best = 0;
		long time = 0;
		for(Result result : results) {
			correct += result.correct;
			if(result.accuracy > best) {
				model = result.model;
				best = result.accuracy;
			}
			time += result.trainingTime + result.testingTime;
		}
		this.model = model;
		this.correct = correct;
		this.accuracy = ((double) correct) / ((double) data.points.size());
		this.time = time;
	}
	
	@Override
	public String toString() {
		String str = "[" + learner + " on " + data.name + "; ";
		str += correct + " correct (" + Utilities.percent(accuracy) + ") in " + Utilities.time(time);
		return str + "]";
	}
}