package edu.uky.ai.sl;

import edu.uky.ai.data.LabeledDataSet;
import edu.uky.ai.util.Utilities;

/**
 * Summarizes the results of a {@link Learner} training a {@link Model} on a
 * given training data set and then classifying the points in a given testing
 * data set. This object represents the results on a single fold when doing
 * k-fold cross-validation.
 * 
 * @author Stephen G. Ware
 */
public class Result {
	
	/** A message indicating success */
	static final String SUCCESS = "success";

	/** The learner used to train the model */
	public final Learner learner;
	
	/** The training data set */
	public final LabeledDataSet training;
	
	/** The model learned */
	public final Model model;
	
	/** The number of milliseconds taken to train the model */
	public final long trainingTime;
	
	/** The testing data set */
	public final LabeledDataSet testing;
	
	/** The number of data points in the testing data set classified correctly */
	public final int correct;
	
	/** The accuracy of the model on the testing data set */
	public final double accuracy;
	
	/** The number of milliseconds taken to test the model */
	public final long testingTime;
	
	/** A short message indicating success or giving details on any errors */
	public final String reason;
	
	/**
	 * Constructs a result object with the given parameters.
	 * 
	 * @param learner the learner
	 * @param training the training data
	 * @param model the model learned
	 * @param trainingTime the number of milliseconds taken to traing the model
	 * @param testing the testing data
	 * @param correct the number of test points correctly classified
	 * @param testingTime the number of milliseconds taken to test the model
	 * @param reason a short message
	 */
	public Result(Learner learner, LabeledDataSet training, Model model, long trainingTime, LabeledDataSet testing, int correct, long testingTime, String reason) {
		this.learner = learner;
		this.training = training;
		this.model = model;
		this.trainingTime = trainingTime;
		this.testing = testing;
		this.correct = correct;
		this.accuracy = testing.points.size() > 0 ? ((double) correct) / ((double) testing.points.size()) : 0;
		this.testingTime = testingTime;
		this.reason = reason;
	}
	
	@Override
	public String toString() {
		String str = "[" + learner + " ";
		if(model == null)
			str += "failed on " + training.name + "; " + reason;
		else {
			str += "trained on " + training.name + "(" + training.points.size() + ") in " + Utilities.time(trainingTime) + "; ";
			str += "tested on " + testing.name + "(" + testing.points.size() + ") in " + Utilities.time(testingTime) + ", " + correct + " correct (" + Utilities.percent(accuracy) + ")";
			if(!reason.equals(SUCCESS))
				str += "; " + reason;
		}
		return str + "]";
	}
}