package edu.uky.ai.data;

/**
 * A numeric value is any value which can be represented by a number (in this
 * case, Java's double data type). Numeric values are
 * {@link java.lang.Comparable Comparable} but not necessarily
 * {@link Interval}, meaning the values are ordered but the distance between
 * values is not necessarily proportionate to the difference between values.
 * Numeric values include {@link Ordinal} and {@link Interval} data types.
 * 
 * @author Stephen G. Ware
 */
public interface Numeric extends Value, Comparable<Numeric> {

	/**
	 * Returns the number associated with this value.
	 * 
	 * @return a double
	 */
	public double getNumber();
	
	/**
	 * Returns true if this value is ordered before the given other value.
	 * 
	 * @param other the other value, which must be {@link Numeric}
	 * @return true if this value is ordered before, false if this value is the
	 * same as or would be ordered after the given value
	 * @throws UnsupportedOperationException if the other value is not {@link Numeric}
	 */
	public default boolean comesBefore(Numeric other) {
		if(getClass().equals(other.getClass()))
			return getNumber() < other.getNumber();
		else
			throw new UnsupportedOperationException("Cannot compare " + getClass().getSimpleName().toLowerCase() + " values to " + other.getClass().getSimpleName().toLowerCase() + " values.");
	}
	
	/**
	 * Returns true if this value is ordered after the given other value.
	 * 
	 * @param other the other value, which must be {@link Numeric}
	 * @return true if this value is ordered after, false if this value is the
	 * same as or would be ordered before the given value
	 * @throws UnsupportedOperationException if the other value is not {@link Numeric}
	 */
	public default boolean comesAfter(Numeric other) {
		if(getClass().equals(other.getClass()))
			return getNumber() > other.getNumber();
		else
			throw new UnsupportedOperationException("Cannot compare " + getClass().getSimpleName().toLowerCase() + " values to " + other.getClass().getSimpleName().toLowerCase() + " values.");
	}
	
	@Override
	public default int compareTo(Numeric other) {
		if(comesBefore(other))
			return -1;
		else if(comesAfter(other))
			return 1;
		else
			return 0;
	}
}