package edu.uky.ai.data;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

import edu.uky.ai.Settings;
import edu.uky.ai.util.ImmutableArray;

/**
 * A labeled data set is a collection of {@link LabeledDataPoint}s, meaning one
 * feature is the class label whose value is of primary interest.
 * 
 * @author Stephen G. Ware
 */
public class LabeledDataSet extends DataSet {
	
	/**
	 * Reads a data set in from a CSV file.
	 * 
	 * @param file the data set in CSV format
	 * @return a data set object
	 * @throws IOException if an error occurs while reading or if the file is
	 * not formatted correctly
	 */
	public static LabeledDataSet read(File file) throws IOException {
		DataSet unlabeled = DataSet.read(file);
		Feature<?>[] features = new Feature[unlabeled.features.size() - 1];
		for(int i=0; i<features.length; i++)
			features[i] = unlabeled.features.get(i);
		ArrayList<LabeledDataPoint> points = new ArrayList<>(unlabeled.points.size());
		for(DataPoint point : unlabeled.points) {
			Value[] values = new Value[point.values.size() - 1];
			for(int j=0; j<values.length; j++)
				values[j] = point.values.get(j);
			points.add(new LabeledDataPoint(null, new ImmutableArray<>(values), point.values.get(point.values.size() - 1)));
		}
		return new LabeledDataSet(unlabeled.name, new ImmutableArray<>(features), unlabeled.features.get(unlabeled.features.size() - 1), points);
	}
	
	/** Serial version UID */
	private static final long serialVersionUID = Settings.VERSION_UID;
	
	/** The class label */
	public final Feature<?> labels;
	
	/** All {@link LabeledDataPoint}s in this data set */
	public final ImmutableArray<LabeledDataPoint> points;
	
	/**
	 * Creates a new labeled data set with the given name, features, class
	 * label, and data points. The number and type of features given must match
	 * the number and types of features in each data point, and the label of
	 * each data point must match the type of the class label.
	 * 
	 * @param name the name
	 * @param features the features
	 * @param labels the class label
	 * @param points the labeled data points
	 */
	@SuppressWarnings("unchecked")
	public LabeledDataSet(String name, ImmutableArray<Feature<?>> features, Feature<?> labels, Iterable<LabeledDataPoint> points) {
		super(name, features, (Iterable<DataPoint>) (Iterable<?>) points);
		this.labels = labels;
		this.points = (ImmutableArray<LabeledDataPoint>) (ImmutableArray<?>) super.points;
		for(LabeledDataPoint point : this.points)
			if(!labels.type.isAssignableFrom(point.label.getClass()))
				throw new IllegalArgumentException("The label \"" + point.label + "\" is not " + labels.type.getSimpleName().toLowerCase() + ".");
	}
	
	@Override
	@SuppressWarnings("unchecked")
	public LabeledDataSet remove(Feature<?> feature) {
		if(feature.equals(labels))
			throw new UnsupportedOperationException("Cannot remove the class label.");
		return new LabeledDataSet(name, removeFeatureFromFeatures(feature), labels, (Iterable<LabeledDataPoint>) (Iterable<?>) removeFeatureFromPoints(feature));
	}
	
	/**
	 * Removes the class label from the data set and all its data points,
	 * returning an unlabeled {@link DataSet}.
	 * 
	 * @return the unlabeled data set
	 */
	public DataSet removeLabels() {
		ArrayList<DataPoint> points = new ArrayList<>(this.points.size());
		for(LabeledDataPoint point : this.points)
			points.add(point.removeLabel());
		return new DataSet(name, features, points);
	}
}