package edu.uky.ai.data;

import java.io.Serializable;

import edu.uky.ai.Settings;

/**
 * A feature is an individual quality for which each {@link DataPoint} in a
 * {@link DataSet} defines a value (i.e. a column label). Features can be of
 * several possible types, including {@link Nominal}, {@link Ordinal}, and
 * {@link Interval}.
 * 
 * @param <V> the value type of this feature
 * @author Stephen G. Ware
 */
public class Feature<V extends Value> implements Serializable {

	/** Serial version UID */
	private static final long serialVersionUID = Settings.VERSION_UID;
	
	/** The name of this feature (i.e. column label) */
	public final String name;
	
	/** The type of this feature's values */
	public final Class<V> type;
	
	/**
	 * Constructs a new feature with a given name and value type.
	 * 
	 * @param name the name
	 * @param type the type
	 */
	public Feature(String name, Class<V> type) {
		this.name = name;
		this.type = type;
	}
	
	@Override
	public boolean equals(Object other) {
		return getClass().equals(other.getClass()) && name.equals(((Feature<?>) other).name) && type.equals(((Feature<?>) other).type);
	}
	
	@Override
	public int hashCode() {
		return name.hashCode();
	}
	
	@Override
	public String toString() {
		return name;
	}
	
	/**
	 * Returns true if the feature's values are one of the {@link Discrete}
	 * data types (i.e. {@link Nominal} or {@link Ordinal}), false otherwise.
	 * 
	 * @return true if the feature's values are discrete
	 */
	public boolean isDiscrete() {
		return Discrete.class.isAssignableFrom(type);
	}
	
	/**
	 * Returns true if the feature's values are one of the {@link Numeric}
	 * data types, (i.e. {@link Ordinal} or {@link Interval}) false otherwise.
	 * 
	 * @return true if the feature's values are numeric
	 */
	public boolean isNumeric() {
		return Numeric.class.isAssignableFrom(type);
	}
	
	/**
	 * Returns true if the feature's values are the {@link Nominal}
	 * data type, false otherwise.
	 * 
	 * @return true if the feature's values are nominal
	 */
	public boolean isNominal() {
		return Nominal.class.isAssignableFrom(type);
	}
	
	/**
	 * Returns true if the feature's values are the {@link Ordinal}
	 * data type, false otherwise.
	 * 
	 * @return true if the feature's values are ordinal
	 */
	public boolean isOrdinal() {
		return Ordinal.class.isAssignableFrom(type);
	}
	
	/**
	 * Returns true if the feature's values are the {@link Interval}
	 * data type, false otherwise.
	 * 
	 * @return true if the feature's values are interval
	 */
	public boolean isInterval() {
		return Interval.class.isAssignableFrom(type);
	}
}