package edu.uky.ai.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;

/**
 * A directed acyclic graph (or DAG) whose nodes are of a given type.  This
 * data structure is immutable, meaning that methods which would modify it
 * instead return a copy of the object with the changes reflected in that copy.
 * For the sake of efficiency, modifications which would return an invalid DAG
 * (e.g. adding a cycle) return null instead of throwing an exception.
 * 
 * @author Stephen G. Ware
 * @param <T> the type of object used as nodes in this DAG
 */
public class DirectedAcyclicGraph<T> implements Iterable<T> {

	/**
	 * A structure representing a node object and its children.
	 * 
	 * @author Stephen G. Ware
	 */
	private final class Node {
		
		/** The node object */
		public final T element;
		
		/** This node's children */
		public final ImmutableList<T> children;
		
		/**
		 * Constructs a new node objects with the given element and children.
		 * 
		 * @param element the node object
		 * @param children this node's children
		 */
		private Node(T element, ImmutableList<T> children) {
			this.element = element;
			this.children = children;
		}
		
		/**
		 * Constructs a new node object with no children.
		 * 
		 * @param element the node object
		 */
		public Node(T element) {
			this(element, new ImmutableList<>());
		}
		
		/**
		 * Returns a copy of this node with the given child added to its list
		 * of children.
		 * 
		 * @param child the new child
		 * @return a copy of the node with the new child
		 */
		public Node addEdgeTo(T child) {
			return new Node(element, children.add(child));
		}
	}
	
	/**
	 * All the nodes in this graph.
	 */
	private final ImmutableList<Node> nodes;
	
	/**
	 * Constructs a new DAG with the given nodes.
	 * 
	 * @param nodes all the nodes in this graph
	 */
	private DirectedAcyclicGraph(ImmutableList<Node> nodes) {
		this.nodes = nodes;
	}
	
	/**
	 * Constructs a DAG with no nodes.
	 */
	public DirectedAcyclicGraph() {
		this(new ImmutableList<Node>());
	}
	
	@Override
	public String toString() {
		String str = "";
		boolean first = true;
		for(DirectedEdge<T> edge : edges()) {
			if(first)
				first = false;
			else
				str += ", ";
			str += edge;
		}
		return str;
	}
	
	/**
	 * Returns an {@link Iterable} of all the edges in this graph.
	 * 
	 * @return all edges
	 */
	public Iterable<DirectedEdge<T>> edges() {
		ArrayList<DirectedEdge<T>> edges = new ArrayList<>();
		for(Node node : nodes)
			for(T child : node.children)
				edges.add(new DirectedEdge<T>(node.element, child));
		return edges;
	}
	
	/**
	 * Returns the node associated with this object.
	 * 
	 * @param object the node object
	 * @return the node, or null if there is no node for this object
	 */
	private final Node getNode(T object) {
		ImmutableList<Node> current = nodes;
		while(current != null && current.size() != 0) {
			if(current.first.element == object)
				return current.first;
			current = current.rest;
		}
		return null;
	}
	
	/**
	 * Returns a new DAG with a new edge from 'before' to 'after.'  If either
	 * node does not exist, they will exist in the new DAG.
	 * 
	 * @param before the tail of the new directed edge
	 * @param after the head of the new directed edge
	 * @return a new DAG with this edge, or null if this edge would create a cycle
	 */
	public DirectedAcyclicGraph<T> add(T before, T after) {
		DirectedAcyclicGraph<T> result = this;
		Node beforeNode = getNode(before);
		if(beforeNode == null) {
			beforeNode = new Node(before);
			result = new DirectedAcyclicGraph<T>(result.nodes.add(beforeNode));
		}
		Node afterNode = getNode(after);
		if(afterNode == null) {
			afterNode = new Node(after);
			result = new DirectedAcyclicGraph<T>(result.nodes.add(afterNode));
		}
		// Would this ordering create a cycle?
		if(path(afterNode, before))
			return null;
		// Does this ordering already exist?
		if(path(beforeNode, after))
			return this;		
		beforeNode = beforeNode.addEdgeTo(after);
		result = new DirectedAcyclicGraph<T>(replace(beforeNode, result.nodes));
		return result;
	}
	
	/**
	 * Replaces one version of a node object with another version.
	 * 
	 * @param node the replacement node
	 * @param nodes the list of nodes in which the original will be replaced
	 * @return a list which contains the new node object
	 */
	private final ImmutableList<Node> replace(Node node, ImmutableList<Node> nodes) {
		if(nodes.first.element == node.element)
			return nodes.rest.add(node);
		else
			return replace(node, nodes.rest).add(nodes.first);
	}
	
	/**
	 * Checks whether or not a path exists from one node to another in this
	 * DAG.
	 * 
	 * @param before the node at which the path should start
	 * @param after the node at which the path should end
	 * @return true if such a path was found, false otherwise
	 */
	public boolean path(T before, T after) {
		Node start = getNode(before);
		if(start == null)
			return false;
		else
			return path(start, after);
	}
	
	/**
	 * Helper method for finding paths.
	 * 
	 * @param from the current node
	 * @param to the destination node
	 * @return true if a path is found, false otherwise
	 */
	private final boolean path(Node from, T to) {
		if(from.element == to)
			return true;
		ImmutableList<T> children = from.children;
		while(children.size() != 0) {
			if(path(getNode(children.first), to))
				return true;
			children = children.rest;
		}
		return false;
	}

	@Override
	public Iterator<T> iterator() {
		return topologicalSort().iterator();
	}
	
	/**
	 * Performs a topological sort of the nodes in this graph and returns a
	 * list of nodes in sorted order.  A topological sort means that a node
	 * always appears before any of its children.
	 * 
	 * @return nodes in topological sorted order
	 */
	private final LinkedList<T> topologicalSort() {
		LinkedList<Node> nodes = new LinkedList<>();
		HashMap<T, Integer> inDegree = new HashMap<>();
		for(Node node : this.nodes)
			inDegree.put(node.element, 0);
		for(Node node : this.nodes)
			for(T child : node.children)
				inDegree.put(child, inDegree.get(child) + 1);
		while(nodes.size() < this.nodes.size()) {
			Node start = null;
			for(T element : inDegree.keySet()) {
				if(inDegree.get(element) == 0) {
					start = getNode(element);
					if(!nodes.contains(start))
						break;
				}
			}
			dfs(start, nodes);
		}
		LinkedList<T> steps = new LinkedList<>();
		for(Node node : nodes)
			steps.add(node.element);
		return steps;
	}
	
	/**
	 * A helper method for performing topological sort which performs a depth
	 * first traversal of the graph, adding each node to the list once all its
	 * children have been visited.
	 * 
	 * @param node the current node
	 * @param nodes the list of searched nodes
	 */
	private final void dfs(Node node, LinkedList<Node> nodes) {
		if(nodes.contains(node))
			return;
		ImmutableList<T> children = node.children;
		while(children.size() != 0) {
			dfs(getNode(children.first), nodes);
			children = children.rest;
		}
		nodes.add(0, node);
	}
}
