package edu.uky.ai.rl.dungeon;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.util.HashMap;

import javax.imageio.ImageIO;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;

import edu.uky.ai.Settings;
import edu.uky.ai.rl.Action;
import edu.uky.ai.rl.ProcessPanel;
import edu.uky.ai.rl.Transition;

/**
 * A visualization of a {@link Dungeon}.
 * 
 * @author Stephen G. Ware
 */
public class DungeonGUI extends ProcessPanel {

	private static final long serialVersionUID = Settings.VERSION_UID;
	private static final int SPRITE_WIDTH = 24;
	private static final int SPRITE_HEIGHT = 24;
	private static final int ZOOM = 2;
	private static final HashMap<String, BufferedImage> SPRITES = new HashMap<>();
	private final Dungeon dungeon;
	private final JTextArea transcript = new JTextArea();
	
	DungeonGUI(int delay, Dungeon dungeon) throws IOException {
		super(delay);
		setLayout(new BorderLayout());
		this.dungeon = dungeon;
		if(SPRITES.size() == 0) {
			BufferedImage sheet = ImageIO.read(DungeonGUI.class.getResourceAsStream("sprites.png"));
			String[] keys = new String[] { Tile.UNKNOWN, Tile.ENTERANCE, Tile.EXIT, Tile.EMPTY, Tile.WALL, Tile.ALIVE_RAT, Tile.ALIVE_GOBLIN, Tile.ALIVE_DRAGON, Tile.CLOSED_CHEST, Tile.OPEN_CHEST, Tile.ALIVE_PLAYER, Tile.DEAD_PLAYER };
			for(int i=0; i<keys.length; i++)
				SPRITES.put(keys[i], sheet.getSubimage(i * SPRITE_WIDTH, 0, SPRITE_WIDTH, SPRITE_HEIGHT));
			SPRITES.put(Tile.DEAD_RAT, SPRITES.get(Tile.DEAD_PLAYER));
			SPRITES.put(Tile.DEAD_GOBLIN, SPRITES.get(Tile.DEAD_PLAYER));
			SPRITES.put(Tile.DEAD_DRAGON, SPRITES.get(Tile.DEAD_PLAYER));
		}
		add(dungeonPanel, BorderLayout.CENTER);
		transcript.setFont(new Font("monospaced", Font.BOLD, 14));
		transcript.setBorder(null);
		transcript.setColumns(40);
		transcript.setLineWrap(true);
		transcript.setWrapStyleWord(true);
		transcript.setBackground(Color.white);
		transcript.setForeground(Color.black);
		transcript.setEditable(false);
		scroll.setBorder(null);
		add(scroll, BorderLayout.EAST);
	}
	
	@Override
	protected void doUpdate(Action action, Transition result) {
		transcript.append("> " + action + "\n" + result.toString() + "\n");
		transcript.setCaretPosition(transcript.getDocument().getLength() - 1);
		repaint();
	}
	
	private final JPanel dungeonPanel = new JPanel() {

		private static final long serialVersionUID = 1L;
		
		@Override
		public Dimension getPreferredSize() {
			return new Dimension(dungeon.getWidth() * SPRITE_WIDTH * ZOOM, dungeon.getHeight() * SPRITE_HEIGHT * ZOOM);
		}
		
		@Override
		public Dimension getMinimumSize() {
			return getPreferredSize();
		}
		
		@Override
		public Dimension getMaximumSize() {
			return getPreferredSize();
		}
		
		@Override
		public void paintComponent(Graphics g) {
			super.paintComponent(g);
			Graphics2D g2d = (Graphics2D) g;
			for(int x=0; x<dungeon.getHeight(); x++) {
				for(int y=0; y<dungeon.getWidth(); y++) {
					BufferedImage sprite = SPRITES.get(dungeon.toString(x, y));
					g2d.drawImage(sprite, y * SPRITE_WIDTH * ZOOM, x * SPRITE_HEIGHT * ZOOM, SPRITE_WIDTH * ZOOM, SPRITE_HEIGHT * ZOOM, null);
				}
			}
		}
	};
	
	private final JScrollPane scroll = new JScrollPane(transcript) {
		
		private static final long serialVersionUID = 1L;

		@Override
		public Dimension getPreferredSize() {
			return new Dimension(super.getPreferredSize().width, dungeonPanel.getPreferredSize().height);
		}
		
		@Override
		public Dimension getMinimumSize() {
			return getPreferredSize();
		}
		
		@Override
		public Dimension getMaximumSize() {
			return getPreferredSize();
		}
	};
}
