package edu.uky.ai.rl.dungeon;

import java.io.IOException;
import java.util.Random;

import edu.uky.ai.rl.Action;
import edu.uky.ai.rl.State;
import edu.uky.ai.rl.StochasticProcess;
import edu.uky.ai.util.ImmutableArray;

/**
 * Represents a specific dungeon which contains monsters and treasure chests
 * and which can be explored for treasure.
 * 
 * @author Stephen G. Ware
 */
public class Dungeon extends StochasticProcess {
	
	private final Tile[][] map;
	private final Random random;
	private final boolean[][] visibility;
	private DungeonState current;
	private DungeonGUI gui = null;
	
	private Dungeon(String name, Tile[][] map, DungeonState initial) {
		super(name, initial, DungeonAction.ALL);
		this.map = map;
		this.random = new Random(name.hashCode());
		this.visibility = new boolean[map.length][map[0].length];
		this.current = initial;
		this.visibility[current.x][current.y] = true;
	}
	
	Dungeon(String name, Tile[][] map) {
		this(name, map, initial(map));
	}
	
	private static final DungeonState initial(Tile[][] map) {
		int px = -1;
		int py = -1;
		int stateful = 0;
		for(int x=0; x<map.length; x++) {
			for(int y=0; y<map[0].length; y++) {
				if(map[x][y] instanceof Tile.StatefulTile)
					stateful++;
				else if(map[x][y] instanceof Tile.Enterance) {
					px = x;
					py = y;
				}
			}
		}
		Integer[] tiles = new Integer[stateful];
		for(int i=0; i<tiles.length; i++)
			tiles[i] = 0;
		return new DungeonState(px, py, new ImmutableArray<>(tiles), false);
	}
	
	@Override
	public Dungeon clone() {
		return new Dungeon(name, map);
	}

	@Override
	public DungeonTransition transition(State state, Action action) {
		DungeonState dstate = (DungeonState) state;
		DungeonAction daction = (DungeonAction) action;
		int x = dstate.x;
		int y = dstate.y;
		switch(daction) {
		case UP: x--; break;
		case DOWN: x++; break;
		case LEFT: y--; break;
		case RIGHT: y++; break;
		}
		visibility[x][y] = true;
		DungeonTransition result = map[x][y].moveTo(dstate, random);
		current = result.state;
		if(gui != null)
			gui.update(daction, result);
		return result;
	}

	@Override
	public DungeonGUI getGUI(int delay) {
		if(gui == null || gui.delay != delay) {
			try {
				gui = new DungeonGUI(delay, this);
			}
			catch(IOException ex) {
				throw new RuntimeException("Failed to load " + getClass() + ".", ex);
			}
		}
		return gui;
	}
	
	int getHeight() {
		return map.length;
	}
	
	int getWidth() {
		return map[0].length;
	}
	
	String toString(int x, int y) {
		if(visibility[x][y])
			return map[x][y].toString(current);
		else
			return Tile.UNKNOWN;
	}
}
