package edu.uky.ai.rl;

import edu.uky.ai.SearchBudget;
import edu.uky.ai.util.ImmutableArray;

/**
 * A stochastic (or random) process is one which
 * {@link #transition(State, Action) transitions} from one {@link State} to
 * another probabilistically based on the {@link Action} taken. A stochastic
 * process is different from a deterministic process because a deterministic
 * process guarantees that the same action taken in the same state will always
 * lead to the same next state. This is not true of a stochastic process. The
 * same action taken in the same state can lead to different states depending
 * on random influence.
 * 
 * This object represents a fully observable but unknown stochastic process.
 * The process is fully observable in the sense that a {@link State} fully
 * describes everything about the process at that moment, and every feature of
 * the state is accurate and trustworthy. However, the process is unknown
 * because the underlying model of how many states exist, the possible values
 * of each feature, and how one state transitions to another is not
 * available in advance (but can be learned).
 * 
 * A stochastic process is what a {@link Learner} creates a {@link Policy} to
 * solve. It's most important method is {@link #transition(State, Action)},
 * which provides, for some given state and action, a valid next state of the
 * process and some reward or punishment that describes how good or bad that
 * transition was. Each call to {@link #transition(State, Action)} counts as
 * one operation for the process's search budget.
 * 
 * @author Stephen G. Ware
 */
public abstract class StochasticProcess implements Cloneable {

	/** The name of the process */
	public final String name;
	
	/** The initial state (how the process starts) */
	public final State initial;
	
	/** The list of available actions that can be taken */
	public final ImmutableArray<Action> actions;
	
	/** Constrains the number of operations and the amount of time this process can use */
	public final SearchBudget budget;
	
	/**
	 * Constructs a new stochastic process with the given parameters.
	 * 
	 * @param name the name of the process
	 * @param initial the initial state of the process
	 * @param actions the list of all possible actions that can be taken
	 * @param budget the number of operations and amount of time this process can use
	 */
	public StochasticProcess(String name, State initial, ImmutableArray<Action> actions, SearchBudget budget) {
		this.name = name;
		this.initial = initial;
		this.actions = actions;
		this.budget = budget;
	}
	
	/**
	 * Constructs a new stochastic process with the given parameters and an
	 * infinite search budget.
	 * 
	 * @param name the name of the process
	 * @param initial the initial state of the process
	 * @param actions the list of all possible actions that can be taken
	 */
	public StochasticProcess(String name, State initial, ImmutableArray<Action> actions) {
		this(name, initial, actions, SearchBudget.INFINITE);
	}
	
	@Override
	public String toString() {
		return name;
	}
	
	@Override
	public abstract StochasticProcess clone();
	
	/**
	 * Given some state for the process and an action to take, this method
	 * returns a {@link Transition}, which describes a valid next state for the
	 * process to be in after taking that action and a reward that indicates
	 * how good or bad the transition was. Only states generated by previous
	 * calls to this function and actions that appear in {@link #actions}
	 * should be used as arguments to this method. Using states or actions
	 * from a different stochastic process object will result in undefined
	 * behavior and probably an exception.
	 * 
	 * @param state a valid state for this process to be in
	 * @param action an action to take from that state
	 * @return a {@link Transition}
	 * @throws IllegalStateException if the given state is a terminal state
	 */
	public abstract Transition transition(State state, Action action);
	
	/**
	 * Returns a visualization of this process. This method is typically only
	 * called from {@link ProcessFrame}.
	 * 
	 * @param delay the number of milliseconds to wait between frames of the visualization
	 * @return a {@link ProcessPanel} to be displayed by a {@link ProcessFrame}
	 */
	public abstract ProcessPanel getGUI(int delay);
}
