 package edu.uky.ai.rl;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;

import edu.uky.ai.rl.dungeon.DungeonParser;

/**
 * Reads different kinds of {@link StochasticProcess}es based on their file
 * extensions.
 * 
 * @author Stephen G. Ware
 */
public class ProcessParser {
	
	/**
	 * Each kind of process which can be parsed should provide one of these to
	 * define how a process can be read from file.
	 * 
	 * @author Stephen G. Ware
	 */
	@FunctionalInterface
	public static interface Parser {
		
		/**
		 * Given some input stream, this method reads the stream and constructs
		 * a {@link StochasticProcess} of a certain type.
		 * 
		 * @param name the name of the stream (usually the nam of the file being read)
		 * @param stream the stream from which to read input
		 * @return a stochastic process
		 * @throws IOException if an exception occurs while reading from the stream
		 */
		public StochasticProcess parse(String name, InputStream stream) throws IOException;
	}

	/** A table of parsers for each kind of file */
	private final HashMap<String, Parser> parsers = new HashMap<>();
	
	/**
	 * Constructs a new parser and populates its table of legal file
	 * extensions.
	 */
	public ProcessParser() {
		parsers.put("dungeon", new DungeonParser());
	}
	
	/**
	 * A convenience method for calling {@link #parse(String, InputStream)}
	 * using a file as the source of input.
	 * 
	 * @param file the file to read from
	 * @return the stochastic process encoded in that file
	 * @throws IOException if an exception occurs while reading the file
	 */
	public StochasticProcess parse(File file) throws IOException {
		try(BufferedInputStream stream = new BufferedInputStream(new FileInputStream(file))) {
			return parse(file.getName(), stream);
		}
	}
	
	/**
	 * Reads and constructs a stochastic process from an input source.
	 * 
	 * @param name the name of the process
	 * @param stream the stream to read
	 * @return a stochastic process
	 * @throws IOException if an exception occurs while reading from the stream
	 */
	public StochasticProcess parse(String name, InputStream stream) throws IOException {
		String extension = "";
		int position = name.lastIndexOf(".");
		if(position > 0) {
			extension = name.substring(position + 1);
			name = name.substring(0, position);
		}
		if(parsers.containsKey(extension))
			return parsers.get(extension).parse(name, stream);
		else
			throw new IllegalArgumentException("The extension \"" + extension + "\" was not recognized.");
	}
}
