package edu.uky.ai.rl;

import javax.swing.JPanel;

/**
 * A visual representation of a {@link StochasticProcess}.
 * 
 * @author Stephen G. Ware
 */
public abstract class ProcessPanel extends JPanel {
	
	private static final long serialVersionUID = edu.uky.ai.Settings.VERSION_UID;
	
	/** The number of milliseconds to wait after drawing a change to the process's state */
	public final int delay;
	
	/**
	 * Creates a new GUI with the given delay.
	 * 
	 * @param delay the delay between updates in milliseconds
	 */
	public ProcessPanel(int delay) {
		this.delay = delay;
	}
	
	/**
	 * Signals that the process this object visualizes has changed via the
	 * given action and resulted in the new state and reward. This method
	 * calls {@link #doUpdate(Action, Transition)} and then waits the given
	 * {@link #delay}.
	 * 
	 * @param action the most recent action taken
	 * @param result the new state and reward that resulted from that action
	 */
	public final void update(Action action, Transition result) {
		doUpdate(action, result);
		repaint();
		try {
			Thread.sleep(delay);
		}
		catch(InterruptedException ex) {
			// do nothing
		}
	}
	
	/**
	 * Signals that the process this object visualizes has changed via the
	 * given action and resulted in the new state and reward. This method
	 * should redraw the GUI accordingly.
	 * 
	 * @param action the most recent action taken
	 * @param result the new state and reward that resulted from that action
	 */
	protected abstract void doUpdate(Action action, Transition result);
}
