package edu.uky.ai.chess.state;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.function.Predicate;

import edu.uky.ai.SearchBudget;

/**
 * Represents the current state (between moves) of a game of chess.
 * 
 * @author Stephen G. Ware
 */
public final class State {
	
	/** The state before the most recent move */
	public final State previous;
	
	/** The current configuration of the board */
	public final Board board;
	
	/** The player whose turn it is to move */
	public final Player player;
	
	/** The current turn number (starting at 0), where a turn includes one move by both players */
	public final int turn;
	
	/** The number of moves remaining until the game is declared a draw */
	public final int movesUntilDraw;
	
	/** Indicates whether or not the player whose turn it is to move is in check */
	public final boolean check;
	
	/** Indicates the outcome of the game (or null if the game has not yet ended) */
	public final Outcome outcome;
	
	/** The search budget, which constrains how many more states may be generated and how much time remains to make a decision */
	public final SearchBudget budget;
	
	/** The number of descendants that have been generated from this state */
	private int descendants = 0;
	
	/**
	 * Constructs a new state based on a given previous state but with a given
	 * board configuration.
	 * 
	 * @param previous the previous state
	 * @param current the current board configuration
	 */
	State(State previous, Board current) {
		this.previous = previous;
		board = current;
		player = previous.player.other();
		turn = previous.player == Player.BLACK ? previous.turn + 1 : previous.turn;
		movesUntilDraw = previous.board.countPieces() > current.countPieces() ? 100 : previous.movesUntilDraw - 1;
		check = board.inCheck(player);
		if(movesUntilDraw == 0 || threepeat(this))
			outcome = Outcome.DRAW;
		else if(next((board) -> { return true; }).isEmpty()) {
			if(check) {
				if(player == Player.WHITE)
					outcome = Outcome.BLACK_WINS;
				else
					outcome = Outcome.WHITE_WINS;
			}
			else
				outcome = Outcome.DRAW;
		}
		else
			outcome = null;
		State parent = previous;
		budget = previous.budget;
		budget.incrementOperations();
		budget.checkTime();
		while(parent != null) {
			parent.descendants++;
			parent = parent.previous;
		}
	}
	
	/**
	 * Tests whether a given board state has occurred three previous times.
	 * 
	 * @param state the state to test
	 * @return true if it has occurred 3 or more times
	 */
	private static final boolean threepeat(State state) {
		int count = 1;
		State current = state.previous;
		while(current != null) {
			if(current.board.equals(state.board))
				count++;
			if(count == 3)
				return true;
			current = current.previous;
		}
		return false;
	}
	
	/**
	 * Constructs a new state object in the initial configuration of a chess
	 * board before a game starts.
	 */
	public State() {
		previous = null;
		board = new Board();
		player = Player.WHITE;
		turn = 0;
		movesUntilDraw = 100;
		check = false;
		outcome = null;
		budget = SearchBudget.INFINITE;
	}
	
	private State(State toClone, SearchBudget budget) {
		previous = toClone.previous;
		board = toClone.board;
		player = toClone.player;
		turn = toClone.turn;
		movesUntilDraw = toClone.movesUntilDraw;
		check = toClone.check;
		outcome = toClone.outcome;
		this.budget = budget;
	}
	
	@Override
	public String toString() {
		if(previous == null)
			return "";
		String str = previous.toString();
		if(!str.isEmpty())
			str += " ";
		if(player == Player.BLACK)
			str += (turn + 1) + ". ";
		return str += PGN.move(previous, this);
	}
	
	/**
	 * Returns a collection of all the possible next states that could result
	 * from the current player making one move in this state.
	 * 
	 * @return the next legal states
	 */
	public final Iterable<State> next() {
		return new StateIterable(next((board) -> {
			return true;
		}));
	}
	
	/**
	 * Returns a collection of all the possible next states that could result
	 * from the current player making one move for a given piece in this state.
	 * 
	 * @param piece the piece to move
	 * @return the next legal states in which the given piece has been moved
	 */
	public final Iterable<State> next(Piece piece) {
		if(!board.contains(piece))
			throw new IllegalArgumentException("No such piece " + piece + PGN.square(piece.file, piece.rank));
		return new StateIterable(next((board) -> {
			return !board.contains(piece);
		}));
	}
	
	/**
	 * Returns the next state that would result from moving a specific piece to
	 * a specific new location.
	 * 
	 * @param from the piece to move at its current location
	 * @param to the piece to move at its new location
	 * @return the state that would result from that move
	 * @throws IllegalArgumentException if the given move is illegal
	 */
	public final State next(Piece from, Piece to) {
		if(!board.contains(from))
			throw new IllegalArgumentException("No such piece " + from + PGN.square(from.file, from.rank));
		ArrayList<Board> next = next((board) -> {
			return !board.contains(from) && board.contains(to);
		});
		if(next.isEmpty())
			throw new IllegalArgumentException("Illegal move " + from + PGN.square(from.file, from.rank) + " to " + PGN.square(to.file, to.rank));
		return new State(this, next.get(0));
	}
	
	/**
	 * Returns a list of the next legal board states that pass a given test.
	 * 
	 * @param predicate the test predicate
	 * @return a list of states passing the test
	 */
	private final ArrayList<Board> next(Predicate<Board> predicate) {
		ArrayList<Board> boards = new ArrayList<>();
		if(outcome != null)
			return boards;
		board.anyMove(player, (board) -> {
			if(board.inCheck(player))
				return false;
			if(predicate.test(board))
				boards.add(board.copy(State.this.board));
			return false;
		});
		return boards;
	}
	
	/**
	 * An {@link java.lang.Iterable} object to wrap around a list of
	 * {@link Board} objects.
	 * 
	 * @author Stephen G. Ware
	 */
	private final class StateIterable implements Iterable<State> {

		/** The boards iterable */
		private final Iterable<Board> boards;
		
		/**
		 * Constructs a new iterable.
		 * 
		 * @param boards the board iterable
		 */
		StateIterable(Iterable<Board> boards) {
			this.boards = boards;
		}
		
		@Override
		public Iterator<State> iterator() {
			return new StateIterator(boards.iterator());
		}
	}
	
	/**
	 * An iterator which converts boards into states.
	 * 
	 * @author Stephen G. Ware
	 */
	private final class StateIterator implements Iterator<State> {

		/** A boards iterator */
		private final Iterator<Board> boards;
		
		/**
		 * Constructs a new state iterator.
		 * 
		 * @param boards a board iterator
		 */
		StateIterator(Iterator<Board> boards) {
			this.boards = boards;
		}
		
		@Override
		public boolean hasNext() {
			return boards.hasNext();
		}

		@Override
		public State next() {
			return new State(State.this, boards.next());
		}
	}
	
	/**
	 * Returns a copy of this state, but with a given search budget.
	 * 
	 * @param budget the new search budget to use
	 * @return a copy of this state with the new search budget
	 */
	public State setSearchBudet(SearchBudget budget) {
		return new State(this, budget);
	}
	
	/**
	 * Returns the number of descendant states which claim this state as an
	 * ancestor.
	 * 
	 * @return the number of descendants
	 */
	public final int countDescendants() {
		return descendants;
	}
}
