package edu.uky.ai.chess.state;

import java.util.function.Predicate;

/**
 * Represents the pawn chess piece.
 * 
 * @author Stephen G. Ware
 */
public class Pawn extends Piece {

	/**
	 * Constructs a new pawn for the given player at the given location.
	 * 
	 * @param player the player who controls the pawn
	 * @param file the piece's file (column) on the board
	 * @param rank the piece's rank (row) on the board
	 */
	public Pawn(Player player, int file, int rank) {
		super(player, file, rank);
	}
	
	@Override
	public String toString() {
		return "P";
	}

	@Override
	boolean anyMove(Board board, Predicate<Board> predicate) {
		int forward = player == Player.WHITE ? 1 : -1;
		int home = player == Player.WHITE ? 1 : 6;
		if(Board.isValid(file, rank + forward) && !board.pieceAt(file, rank + forward)) {
			if(board.tryMove(this, 0, forward, predicate))
				return true;
			if(rank == home && Board.isValid(file, rank + forward * 2) && !board.pieceAt(file, rank + forward * 2))
				if(board.tryMove(this, 0, forward * 2, predicate))
					return true;
		}
		if(Board.isValid(file + 1, rank + forward)) {
			if(board.pieceAt(file + 1, rank + forward, player.other()) && board.tryMove(this, 1, forward, predicate))
				return true;
			if(board.enPassant(this, 1, forward) != -1 && board.tryMove(this, 1, forward, predicate))
				return true;
		}
		if(Board.isValid(file - 1, rank + forward)) {
			if(board.pieceAt(file - 1, rank + forward, player.other()) && board.tryMove(this, -1, forward, predicate))
				return true;
			if(board.enPassant(this, -1, forward) != -1 && board.tryMove(this, -1, forward, predicate))
				return true;
		}
		return false;
	}

	@Override
	public Pawn move(int dfile, int drank) {
		return new Pawn(player, file + dfile, rank + drank);
	}
}
