package edu.uky.ai.chess;

import edu.uky.ai.chess.gui.ChessDisplay;
import edu.uky.ai.chess.state.Outcome;
import edu.uky.ai.chess.state.PGN;
import edu.uky.ai.chess.state.Player;
import edu.uky.ai.chess.state.State;

/**
 * Represents a single chess game between two {@link edu.uky.ai.chess.Agent}s.
 * 
 * @author Stephen G. Ware
 */
public class Game {
	
	/** the maximum moves that may be considered when an agent makes a decision */
	public final int maxMoves;
	
	/** the maximum milliseconds an agent may take to decide on a move */
	public final long maxTime;

	/** The white player */
	public Agent white;
	
	/** The black player */
	public Agent black;
	
	/** The current game state */
	private State state = new State();
	
	/** The exception that caused the game to end prematurely (if any) */
	private Exception exception = null;
	
	/** The outcome of the game (or null if it has not yet ended) */
	private Outcome outcome = null;
	
	/** Number of moves made by white */
	private int whiteMoves = 0;
	
	/** Number of {@link edu.uky.ai.chess.state.State}s generated by white during the entire game */
	private int whiteStates = 0;
	
	/** Number of moves made by black */
	private int blackMoves = 0;
	
	/** Number of {@link edu.uky.ai.chess.state.State}s generated by black during the entire game */
	private int blackStates = 0;
	
	/**
	 * Constructs a new game between the given white and black
	 * {@link edu.uky.ai.chess.Agent}s.
	 * 
	 * @param white the white player
	 * @param black the black player
	 * @param maxMoves the maximum moves that may be considered when an agent
	 * makes a decision
	 * @param maxTime the maximum milliseconds an agent may take to decide on
	 * a move
	 */
	public Game(Agent white, Agent black, int maxMoves, long maxTime) {
		this.maxMoves = maxMoves;
		this.maxTime = maxTime;
		this.white = white;
		this.black = black;
	}
	
	@Override
	public String toString() {
		return getPrefix() + "\n" + PGN.game(state) + getSuffix();
	}
	
	/**
	 * Returns a PGN prefix to be printed at the start of the game.
	 * 
	 * @return the prefix
	 */
	private String getPrefix() {
		return "[White \"" + white + "\"]\n[Black \"" + black + "\"]";
	}
	
	/**
	 * Returns a PGN suffix to be printed at the end of the game.
	 * 
	 * @return the suffix
	 */
	private String getSuffix() { 
		if(outcome == null)
			return (state.player == Player.BLACK ? " " : "") + "*";
		else {
			String suffix = "";
			if(state.outcome != null && state.player == Player.BLACK)
				suffix += "\n";
			else if(state.outcome == null) {
				if(state.player == Player.BLACK)
					suffix += " ";
				suffix += "{ " + (state.player == Player.WHITE ? toString(white) : toString(black)) + " concedes: " + exception.getMessage() + " }\n";
			}
			if(outcome == Outcome.WHITE_WINS)
				suffix += "1-0";
			else if(outcome == Outcome.BLACK_WINS)
				suffix += "0-1";
			else
				suffix += "1/2-1/2";
			return suffix;
		}
	}
	
	/**
	 * Utility method for converting a given agent object into a string.
	 * 
	 * @param agent the agent
	 * @return a string representation of the agent
	 */
	private String toString(Agent agent) {
		if(agent.name.equals("Human")) {
			if(agent == white)
				return "White";
			else
				return "Black";
		}
		else
			return agent.name;
	}
	
	/**
	 * Plays the game until it is over, updating the GUI after each move.
	 */
	public void play() {
		ChessDisplay.getInstance().board.setState(state);
		ChessDisplay.getInstance().banner.setText(getBanner());
		ChessDisplay.getInstance().console.append(getPrefix() + "\n");
		while(outcome == null) {
			try {
				State choice;
				if(state.player == Player.WHITE) {
					choice = white.choose(state, maxMoves, maxTime);
					if(choice != state) {
						whiteMoves++;
						whiteStates += choice.previous.countDescendants();
						ChessDisplay.getInstance().console.append((state.turn + 1) + ". " + PGN.move(state, choice));
					}
				}
				else {
					choice = black.choose(state, maxMoves, maxTime);
					if(choice != state) {
						blackMoves++;
						blackStates += choice.previous.countDescendants();
						ChessDisplay.getInstance().console.append(" " + PGN.move(state, choice) + "\n");
					}
				}
				state = choice;
				outcome = state.outcome;
				ChessDisplay.getInstance().board.setState(state);
			}
			catch(Exception e) {
				e.printStackTrace();
				exception = e;
				if(state.player == Player.WHITE)
					outcome = Outcome.BLACK_WINS;
				else
					outcome = Outcome.WHITE_WINS;
			}
			if(outcome != null)
				ChessDisplay.getInstance().console.append(getSuffix() + "\n\n");
			ChessDisplay.getInstance().banner.setText(getBanner());
		}
	}
	
	/**
	 * Returns the text that should be displayed in the banner over the board
	 * GUI.
	 * 
	 * @return the text
	 */
	private final String getBanner() {
		if(outcome == Outcome.DRAW)
			return "Draw";
		else if(outcome != null)
			return toString(getWinner()) + " wins.";
		else if(state.player == Player.WHITE) {
			if(white.name.equals("Human"))
				return "White's turn";
			else
				return toString(white) + " is thinking...";
		}
		else {
			if(black.name.equals("Human"))
				return "Black's turn";
			else
				return toString(black) + " is thinking...";
		}
	}
	
	/**
	 * Returns the {@link edu.uky.ai.chess.Agent} who won the game.
	 * 
	 * @return the winner or null if the game was a draw or stalemate
	 */
	public Agent getWinner() {
		if(outcome == Outcome.WHITE_WINS)
			return white;
		else if(outcome == Outcome.BLACK_WINS)
			return black;
		else if(outcome == null)
			throw new IllegalStateException("Game is not over");
		else
			return null;
	}
	
	/**
	 * Returns the number of moves made so far by white.
	 * 
	 * @return the number of moves
	 */
	public int getWhiteMoves() {
		return whiteMoves;
	}
	
	/**
	 * Returns the number of {@link edu.uky.ai.chess.state.State}s generated so
	 * far by white.
	 * 
	 * @return the number of states
	 */
	public int getWhiteStates() {
		return whiteStates;
	}
	
	/**
	 * Returns the number of moves made so far by black.
	 * 
	 * @return the number of moves
	 */
	public int getBlackMoves() {
		return blackMoves;
	}
	
	/**
	 * Returns the number of {@link edu.uky.ai.chess.state.State}s generated so
	 * far by black.
	 * 
	 * @return the number of states
	 */
	public int getBlackStates() {
		return blackStates;
	}
}
