package edu.uky.ai.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Collection;
import java.util.Iterator;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

/**
 * Assorted helpful methods.
 * 
 * @author Stephen G. Ware
 */
public class Utilities {
	
	/**
	 * Calculates the base 2 logarithm of a number.
	 * 
	 * @param x the number
	 * @return the base 2 logarithm of x
	 */
	public static final double log2(double x) {
		return Math.log(x) / Math.log(2);
	}
	
	/**
	 * Converts a duration in milliseconds into a formatted string expressing
	 * showing minutes, seconds, and milliseconds.
	 * 
	 * @param time the duration in milliseconds
	 * @return a string in the format "minutes:seconds:milliseconds"
	 */
	public static final String time(long time) {
		int minutes = (int) (time / (1000*60));
		int seconds = (int) (time / 1000) % 60;
		int milliseconds = (int) (time % 1000);
		return String.format("%d:%d:%d", minutes, seconds, milliseconds);
	}
	
	/**
	 * Converts a decimal number into a string expressing percentage.
	 * 
	 * @param value a double between 0 and 1 (inclusive)
	 * @return a string representing a percentage
	 */
	public static final String percent(double value) {
		return String.format("%.0f%%", value * 100);
	}
	
	/**
	 * Adds two number objects together. The objects must be instances of
	 * {@link java.lang.Number} or null (which is treated as 0).
	 * 
	 * @param n1 the first number
	 * @param n2 the second number
	 * @return the sum of the numbers
	 */
	public static final Number add(Object n1, Object n2) {
		return fromBigDecimal(toBigDecimal(n1).add(toBigDecimal(n2)));
	}
	
	/**
	 * Subtracts the second number object from the first. The objects must be
	 * instances of {@link java.lang.Number} or null (which is treated as 0).
	 * 
	 * @param n1 the first number
	 * @param n2 the second number
	 * @return the difference of the numbers
	 */
	public static final Number subtract(Object n1, Object n2) {
		return fromBigDecimal(toBigDecimal(n1).subtract(toBigDecimal(n2)));
	}
	
	/**
	 * Multiplies two number objects together. The objects must be instances of
	 * {@link java.lang.Number} or null (which is treated as 0).
	 * 
	 * @param n1 the first number
	 * @param n2 the second number
	 * @return the product of the numbers
	 */
	public static final Number multiply(Object n1, Object n2) {
		return fromBigDecimal(toBigDecimal(n1).multiply(toBigDecimal(n2)));
	}
	
	/**
	 * Divides the first number object by the second. The objects must be
	 * instances of {@link java.lang.Number} or null (which is treated as 0).
	 * 
	 * @param n1 the first number
	 * @param n2 the second number
	 * @return the quotient of the numbers
	 */
	public static final Number divide(Object n1, Object n2) {
		return fromBigDecimal(toBigDecimal(n1).divide(toBigDecimal(n2), RoundingMode.HALF_UP));
	}
	
	private static final BigDecimal toBigDecimal(Object object) {
		if(object == null)
			return BigDecimal.ZERO;
		else
			return BigDecimal.valueOf(((Number) object).doubleValue());
	}
	
	private static final Number fromBigDecimal(BigDecimal number) {
		if(number.remainder(BigDecimal.ONE).compareTo(BigDecimal.ZERO) == 0)
			return number.intValue();
		else
			return number.doubleValue();
	}	
	
	/**
	 * Efficiently converts any {@link java.lang.Iterable} into an array of its
	 * elements.
	 * 
	 * @param <T> the component type of the array to be returned
	 * @param iterable the group of objects to convert to an array
	 * @param type the component type of the array to be returned
	 * @return an array of the given component type containing the objects
	 */
	@SuppressWarnings("unchecked")
	public static final <T> T[] toArray(Iterable<? extends T> iterable, Class<T> type) {
		if(iterable instanceof Collection<?>) {
			Collection<? extends T> collection = (Collection<? extends T>) iterable;
			return collection.toArray((T[]) Array.newInstance(type, collection.size()));
		}
		else
			return toArray(iterable.iterator(), 0, type);
	}
	
	@SuppressWarnings("unchecked")
	private static final <T> T[] toArray(Iterator<? extends T> iterator, int index, Class<T> type) {
		if(iterator.hasNext()) {
			T element = iterator.next();
			T[] array = toArray(iterator, index + 1, type);
			array[index] = element;
			return array;
		}
		else
			return (T[]) Array.newInstance(type, index);
	}

	/**
	 * Returns the name of a file (with no preceding URL information and no
	 * trailing file extensions).
	 * 
	 * @param file the file whose name is needed
	 * @return the file's name
	 */
	public static final String getFileName(File file) {
		String name = file.toURI().toString();
		if(name.lastIndexOf("/") != -1)
			if(name.endsWith("/"))
				name = name.substring(0, name.length() - 1);
			name = name.substring(name.lastIndexOf("/") + 1);
		if(name.indexOf(".") != -1)
			name = name.substring(0, name.indexOf("."));
		return name;
	}
	
	/**
	 * Searches a given JAR file for the definition of a non-abstract class
	 * that extends the given class and returns a new instance of that class.
	 * 
	 * If no such class or more than one such class is found, an exception is
	 * thrown.
	 * 
	 * The class must define a public constructor which taken no arguments.
	 * 
	 * @param <T> an ancestor type of the object being searched for 
	 * @param type an ancestor class of the object being searched for
	 * @param jarFile the JAR file to search
	 * @return an instance of the class which extends the given parent class
	 * @throws IOException if a problem occurs while reading the JAR file
	 * @throws ClassNotFoundException if no such classes or more than one such
	 * classes are found
	 * @throws NoSuchMethodException if the class does not define a public
	 * constructor that takes no arguments
	 * @throws SecurityException if this process violates a security policy
	 * @throws InstantiationException if invoking the constructor causes an
	 * InstantiationException
	 * @throws IllegalAccessException if invoking the constructor causes an
	 * IllegalAccessException
	 * @throws IllegalArgumentException if invoking the constructor causes an
	 * IllegalArgumentException
	 * @throws InvocationTargetException if invoking the constructor causes an
	 * InvocationTargetException
	 */
	@SuppressWarnings("unchecked")
	public static final <T> T loadFromJARFile(Class<T> type, File jarFile) throws IOException, ClassNotFoundException, NoSuchMethodException, SecurityException, InstantiationException, IllegalAccessException, IllegalArgumentException, InvocationTargetException {
		T plugin = null;
		ClassLoader loader = URLClassLoader.newInstance(new URL[]{ jarFile.toURI().toURL() }, Utilities.class.getClassLoader());
		try(ZipInputStream zip = new ZipInputStream(new FileInputStream(jarFile))) {
			for(ZipEntry entry = zip.getNextEntry(); entry != null; entry = zip.getNextEntry()) {
				if(entry.getName().endsWith(".class") && !entry.isDirectory()) {
					StringBuilder className = new StringBuilder();
					for(String part : entry.getName().split("/")) {
						if(className.length() != 0)
							className.append(".");
						className.append(part);
						if(part.endsWith(".class"))
							className.setLength(className.length() - ".class".length());
					}
					Class<?> c = Class.forName(className.toString(), true, loader);
					if(type.isAssignableFrom(c) && !Modifier.isAbstract(c.getModifiers())) {
						if(plugin != null)
							throw new ClassNotFoundException("Both " + plugin.getClass().getName() + " and " + c.getName() + " are subclasses of " + type.getName() + ".");
						Constructor<? extends T> constructor = ((Class<? extends T>) c).getConstructor();
						plugin = constructor.newInstance();
					}
				}
			}
		}
		if(plugin == null)
			throw new ClassNotFoundException("No subclass of " + type + " found in " + jarFile + ".");
		else
			return plugin;
	}
}
