package edu.uky.ai.chess.state;

import java.util.Iterator;
import java.util.function.Predicate;

/**
 * Represents a chess board and the pieces on it.
 * 
 * @author Stephen G. Ware
 */
public class Board implements Cloneable, Iterable<Piece> {

	/**
	 * Tests whether or not a given file and rank describe a square on a chess
	 * board (i.e. both are &gt;= 0 and &lt; 8).
	 * 
	 * @param file the file to test
	 * @param rank the rank to test
	 * @return true if the square is valid, false otherwise
	 */
	public static final boolean isValid(int file, int rank) {
		return file >= 0 &&
				file < 8 &&
				rank >= 0 &&
				rank < 8;
	}
	
	/** The previous board (used for determining things like en passant and castling) */
	final Board previous;
	
	/** An array of {@link edu.uky.ai.chess.state.Piece}s on the board */
	private final Piece[] pieces;
	
	/** The number of pieces on the board */
	private int size;

	/**
	 * Constructs a new board by cloning a given board object.
	 * 
	 * @param previous the previous state of the board to clone
	 * @param toClone the board to clone
	 */
	private Board(Board previous, Board toClone) {
		this.previous = previous;
		pieces = new Piece[toClone.size];
		size = toClone.size;
		int i = 0;
		for(Piece piece : toClone.pieces) {
			if(piece != null) {
				pieces[i] = piece;
				i++;
			}
		}
		
	}
	
	/** The initial configuration of pieces on the board */
	private static final Piece[] INITIAL = new Piece[] {
		new Pawn(Player.WHITE, 0, 1),
		new Pawn(Player.WHITE, 1, 1),
		new Pawn(Player.WHITE, 2, 1),
		new Pawn(Player.WHITE, 3, 1),
		new Pawn(Player.WHITE, 4, 1),
		new Pawn(Player.WHITE, 5, 1),
		new Pawn(Player.WHITE, 6, 1),
		new Pawn(Player.WHITE, 7, 1),
		new Knight(Player.WHITE, 1, 0),
		new Knight(Player.WHITE, 6, 0),
		new Bishop(Player.WHITE, 2, 0),
		new Bishop(Player.WHITE, 5, 0),
		new Queen(Player.WHITE, 3, 0),
		new Rook(Player.WHITE, 0, 0),
		new Rook(Player.WHITE, 7, 0),
		new King(Player.WHITE, 4, 0),
		new Pawn(Player.BLACK, 0, 6),
		new Pawn(Player.BLACK, 1, 6),
		new Pawn(Player.BLACK, 2, 6),
		new Pawn(Player.BLACK, 3, 6),
		new Pawn(Player.BLACK, 4, 6),
		new Pawn(Player.BLACK, 5, 6),
		new Pawn(Player.BLACK, 6, 6),
		new Pawn(Player.BLACK, 7, 6),
		new Knight(Player.BLACK, 1, 7),
		new Knight(Player.BLACK, 6, 7),
		new Bishop(Player.BLACK, 2, 7),
		new Bishop(Player.BLACK, 5, 7),
		new Queen(Player.BLACK, 3, 7),
		new Rook(Player.BLACK, 0, 7),
		new Rook(Player.BLACK, 7, 7),
		new King(Player.BLACK, 4, 7)
	};
	
	/**
	 * Constructs a new chess board in the initial state.
	 */
	Board() {
		previous = null;
		pieces = INITIAL.clone();
		size = INITIAL.length;
	}
	
	@Override
	public boolean equals(Object other) {
		if(other instanceof Board) {
			Board otherBoard = (Board) other;
			if(hashCode() != otherBoard.hashCode())
				return false;
			if(countPieces() != otherBoard.countPieces())
				return false;
			for(Piece piece : otherBoard)
				if(!pieceAt(piece.file, piece.rank, piece.player, piece.getClass()))
					return false;
			return true;
		}
		return false;
	}
	
	/** Stores the hash code after it has been computed */
	private int hashCode = 0;
	
	@Override
	public int hashCode() {
		if(hashCode == 0) {
			for(Piece piece : pieces)
				if(piece != null)
					hashCode += piece.hashCode();
		}
		return hashCode;
	}
	
	@Override
	public String toString() {
		String str = "";
		for(int rank=7; rank>=0; rank--) {
			str += "\n|";
			for(int file=0; file<8; file++) {
				Piece piece = getPieceAt(file, rank);
				if(piece == null)
					str += "  |";
				else {
					str += piece.player == Player.WHITE ? "W" : "B";
					str += piece + "|";
				}
			}
		}
		return str.substring(1);
	}
	
	@Override
	public Board clone() {
		return new Board(previous, this);
	}

	@Override
	public Iterator<Piece> iterator() {
		return new PieceIterator();
	}
	
	/**
	 * Iterates through all the pieces on a board.
	 * 
	 * @author Stephen G. Ware
	 */
	private final class PieceIterator implements Iterator<Piece> {

		/** The current index in the {@link edu.uky.ai.chess.state.Board#pieces} array */
		private int index = -1;
		
		/**
		 * Constructs a new piece iterator.
		 */
		private PieceIterator() {
			advance();
		}
		
		/**
		 * Advances the index to the next piece which has not yet been
		 * returned.
		 */
		private final void advance() {
			do {
				index++;
			} while(index < pieces.length && pieces[index] == null);
		}
		
		@Override
		public boolean hasNext() {
			return index < pieces.length;
		}

		@Override
		public Piece next() {
			Piece piece = pieces[index];
			advance();
			return piece;
		}
	}
	
	/**
	 * Returns a new board object in the same configuration with the given
	 * previous state.
	 * 
	 * @param previous the previous configuration of the board
	 * @return a new board configured like this board object
	 */
	Board copy(Board previous) {
		return new Board(previous, this);
	}
	
	/**
	 * Returns the number of pieces on the board for both players.
	 * 
	 * @return the number of pieces
	 */
	public int countPieces() {
		return size;
	}
	
	/**
	 * Returns the number of pieces on the board for a given player.
	 * 
	 * @param player the player whose pieces should be counted
	 * @return the number of pieces
	 */
	public int countPieces(Player player) {
		int count = 0;
		for(Piece piece : pieces)
			if(piece != null && piece.player == player)
				count++;
		return count;
	}

	/**
	 * Tests whether or not there is a piece at the given location.
	 * 
	 * @param file the file to test
	 * @param rank the rank to test
	 * @return true if any piece is at this location, false otherwise
	 */
	public boolean pieceAt(int file, int rank) {
		return getPieceAt(file, rank) != null;
	}
	
	/**
	 * Tests whether or not there is a piece at the given location for a given
	 * player.
	 * 
	 * @param file the file to test
	 * @param rank the rank to test
	 * @param player the player the piece should belong to
	 * @return true if such a piece is at this location, false otherwise
	 */
	public boolean pieceAt(int file, int rank, Player player) {
		Piece piece = getPieceAt(file, rank);
		return piece != null && piece.player == player;
	}
	
	/**
	 * Tests whether or not there is a piece of a given type at the given
	 * location for a given player
	 * 
	 * @param file the file to test
	 * @param rank the rank to test
	 * @param player the player the piece should belong to
	 * @param type the type the piece should be
	 * @return true if such a piece is at this location, false otherwise
	 */
	public boolean pieceAt(int file, int rank, Player player, Class<? extends Piece> type) {
		Piece piece = getPieceAt(file, rank);
		return piece != null && piece.player == player && piece.getClass() == type;
	}
	
	/**
	 * Tests if a piece of this same type and for the same player exists at the
	 * the given piece's file and rank.
	 * 
	 * @param piece the piece to test for
	 * @return true if such a piece is at this location, false otherwise
	 */
	public boolean contains(Piece piece) {
		return pieceAt(piece.file, piece.rank, piece.player, piece.getClass());
	}
	
	/**
	 * Returns the piece at a given location.
	 * 
	 * @param file the file of the piece
	 * @param rank the rank of the piece
	 * @return the piece, or null if no piece is at that location
	 */
	public Piece getPieceAt(int file, int rank) {
		for(Piece piece : pieces)
			if(piece != null && piece.file == file && piece.rank == rank)
				return piece;
		return null;
	}
	
	/**
	 * Returns the {@link edu.uky.ai.chess.state.King} for the
	 * given player.
	 * 
	 * @param player the player
	 * @return that player's king
	 */
	public King getKing(Player player) {
		for(Piece piece : pieces)
			if(piece != null && piece instanceof King && piece.player == player)
				return (King) piece;
		return null;
	}
	
	/**
	 * A structure to describe the result of a move.
	 * 
	 * @author Stephen G. Ware
	 */
	static class MoveResult {
		
		/** Indicates if the move is legal */
		public final boolean valid;
		
		/** Indicates if the move captured an enemy piece */
		public final boolean capture;
		
		/** Indicates the return value of the test {@link java.util.function.Predicate} */
		public final boolean test;
		
		/**
		 * Constructs a new move result.
		 * 
		 * @param valid indicates if the move is legal
		 * @param capture indicates if the move captured an enemy piece
		 * @param test indicates the return value of the test predicate
		 */
		private MoveResult(boolean valid, boolean capture, boolean test) {
			this.valid = valid;
			this.capture = capture;
			this.test = test;
		}
	}
	
	/** Returned when a move is illegal */
	static final MoveResult INVALID = new MoveResult(false, false, false);
	
	/** Returned when a move does not capture and the test predicate returns false */
	static final MoveResult EMPTY_FALSE = new MoveResult(true, false, false);
	
	/** Returned when a move does not capture and the test predicate returns true */
	static final MoveResult EMPTY_TRUE = new MoveResult(true, false, true);
	
	/** Returned when a move captures and the test predicate returns false */
	static final MoveResult CAPTURE_FALSE = new MoveResult(true, true, false);
	
	/** Returned when a move captures and the test predicate returns true */
	static final MoveResult CAPTURE_TRUE = new MoveResult(true, true, true);
	
	/**
	 * Temporarily modifies the state of this board to reflect the changes made
	 * by a proposed move and returns a {@link MoveResult} to indicate
	 * information about the move and the return value of the test predicate.
	 * Before this method returns, it restores this board to its original
	 * state. 
	 * 
	 * @param piece the piece to move
	 * @param dfile the change in the piece's file
	 * @param drank the change in the piece's rank
	 * @param predicate a predicate which will be used to test the resulting board state(s)
	 * @return the appropriate {@link MoveResult}
	 */
	private final MoveResult _tryMove(Piece piece, int dfile, int drank, Predicate<Board> predicate) {
		if(!isValid(piece.file + dfile, piece.rank + drank))
			return INVALID;
		Piece moved = piece.move(dfile, drank);
		int movedIndex = -1;
		Piece capturedPiece = null;
		int captureIndex = enPassant(piece, dfile, drank);
		if(captureIndex == -1) {
			for(int i=0; i<pieces.length; i++) {
				if(pieces[i] != null && pieces[i].file == moved.file && pieces[i].rank == moved.rank) {
					if(pieces[i].player == piece.player)
						return INVALID;
					captureIndex = i;
					break;
				}
			}
		}
		if(captureIndex != -1) {
			capturedPiece = pieces[captureIndex];
			pieces[captureIndex] = null;
			size--;
		}
		for(int i=0; i<pieces.length; i++) {
			if(pieces[i] == piece) {
				pieces[i] = moved;
				movedIndex = i;
				break;
			}
		}
		Piece castle = null;
		int castleIndex = -1;
		if(piece.getClass() == King.class && Math.abs(dfile) == 2) {
			for(int i=0; i<pieces.length; i++) {
				if(pieces[i] != null && pieces[i].getClass() == Rook.class && pieces[i].player == piece.player) {
					if((dfile == 2 && pieces[i].file == 7) || (dfile == -2 && pieces[i].file == 0)) {
						castle = pieces[i];
						castleIndex = i;
						break;
					}
				}
			}
			if(castle.file == 0)
				pieces[castleIndex] = castle.move(3, 0);
			else
				pieces[castleIndex] = castle.move(-2, 0);
		}
		boolean result;
		if(piece.getClass() == Pawn.class && ((moved.player == Player.WHITE && moved.rank == 7) || (moved.player == Player.BLACK && moved.rank == 0))) {
			pieces[movedIndex] = new Knight(piece.player, moved.file, moved.rank);
			result = predicate.test(this);
			if(!result) {
				pieces[movedIndex] = new Bishop(piece.player, moved.file, moved.rank);
				result = predicate.test(this); 
			}
			if(!result) {
				pieces[movedIndex] = new Rook(piece.player, moved.file, moved.rank);
				result = predicate.test(this); 
			}
			if(!result) {
				pieces[movedIndex] = new Queen(piece.player, moved.file, moved.rank);
				result = predicate.test(this); 
			}
			result = predicate.test(this);
		}
		else
			result = predicate.test(this);
		pieces[movedIndex] = piece;
		if(capturedPiece != null) {
			pieces[captureIndex] = capturedPiece;
			size++;
		}
		if(castle != null)
			pieces[castleIndex] = castle;
		if(result) {
			if(capturedPiece == null)
				return EMPTY_TRUE;
			else
				return CAPTURE_TRUE;
		}
		else {
			if(capturedPiece == null)
				return EMPTY_FALSE;
			else
				return CAPTURE_FALSE;
		}
	}
	
	/**
	 * Tests whether or not a given piece can perform an en passant capture.
	 * 
	 * @param piece the piece to test
	 * @param dfile the change in the piece's file
	 * @param drank the change in the piece's rank
	 * @return the index of the piece that can be captured, or -1 if a capture cannot be performed
	 */
	final int enPassant(Piece piece, int dfile, int drank) {
		if(dfile == 0 || drank == 0 || piece.getClass() != Pawn.class)
			return -1;
		if(pieceAt(piece.file + dfile, piece.rank, piece.player.other(), Pawn.class) &&
				!pieceAt(piece.file + dfile, piece.rank + drank, piece.player.other()) &&
				!pieceAt(piece.file + dfile, piece.rank + (drank * 2), piece.player.other()) &&
				previous != null &&
				previous.pieceAt(piece.file + dfile, piece.rank + (drank * 2), piece.player.other(), Pawn.class))
			for(int i=0; i<pieces.length; i++)
				if(pieces[i] != null && pieces[i].file == piece.file + dfile && pieces[i].rank == piece.rank)
					return i;
		return -1;
	}
	
	/**
	 * Tests whether or not a given piece has moved during the history of the
	 * game.
	 * 
	 * @param piece the piece to check
	 * @return true if it has ever moved, false otherwise
	 * @throws IllegalArgumentException if the given piece does not exist on this board
	 */
	public final boolean hasMoved(Piece piece) {
		Board current = previous;
		while(current != null) {
			Piece other = current.getPieceAt(piece.file, piece.rank);
			if(other == null || !other.equals(piece))
				return true;
			current = current.previous;
		}
		return false;
	}
	
	/**
	 * Temporarily modifies the state of this board to reflect the changes made
	 * by a proposed move, passes itself to a test predicate, restores its
	 * original state, and then returns the result of the predicate.  In the
	 * case of pawn promotion, this object may be modified and passed to the
	 * predicate multiple times. If the predicate ever returns true, no other
	 * states will be tested.
	 * 
	 * @param piece the piece to move
	 * @param dfile the change in the piece's file
	 * @param drank the change in the piece's rank
	 * @param predicate the predicate whose value will be returned
	 * @return the return value of the given predicate when tested on the modified board
	 */
	final boolean tryMove(Piece piece, int dfile, int drank, Predicate<Board> predicate) {
		return _tryMove(piece, dfile, drank, predicate).test;
	}
	
	/**
	 * Calls {@link #tryMove(Piece, int, int, Predicate) tryMove} for every
	 * legal move along a straight line.  If the predicate ever returns true,
	 * no additional states will be tested.
	 * 
	 * @param piece the piece to move in a line
	 * @param dfile the unit change in the piece's file at each step
	 * @param drank the unit change in the piece's rank at each step
	 * @param predicate the predicate to test
	 * @return the return value of the predicate
	 */
	final boolean tryAllMoves(Piece piece, int dfile, int drank, Predicate<Board> predicate) {
		MoveResult result = EMPTY_FALSE;
		for(int i = 1; i < 8 && result.valid && !result.capture; i++) {
			result = _tryMove(piece, dfile * i, drank * i, predicate);
			if(result.test)
				return true;
		}
		return false;
	}
	
	/**
	 * Given a predicate, this method will pass it every possible board
	 * configuration that could result from a single move by the given player.
	 * Note that this may include some illegal moves (such as moving into
	 * check).  If the predicate ever returns true, no additional moves will
	 * be tested and this method will return true.
	 * 
	 * @param player the player to move
	 * @param predicate the predicate to test
	 * @return the return value of the predicate
	 */
	final boolean anyMove(Player player, Predicate<Board> predicate) {
		for(Piece piece : pieces)
			if(piece != null && piece.player == player)
				if(piece.anyMove(this, predicate))
					return true;
		return false;
	}
	
	/**
	 * Tests whether a given player is in check.
	 * 
	 * @param player the player to test
	 * @return true if the player is in check, false otherwise
	 */
	final boolean inCheck(Player player) {
		King king = getKing(player);
		if(king == null)
			return true;
		for(int i=0; i<pieces.length; i++)
			if(pieces[i] != null && threatens(pieces[i], king))
				return true;
		return false;
	}
	
	/**
	 * Tests whether a given piece threatens the given king.
	 * 
	 * @param piece the piece to test
	 * @param king the king
	 * @return true if the piece threatens the king, false otherwise
	 */
	private final boolean threatens(Piece piece, King king) {
		if(piece.player == king.player)
			return false;
		else if(piece.getClass() == Pawn.class)
			return pawnThreatens(piece, king);
		else if(piece.getClass() == Knight.class)
			return knightThreatens(piece, king);
		else if(piece.getClass() == Bishop.class)
			return bishopThreatens(piece, king);
		else if(piece.getClass() == Rook.class)
			return rookThreatens(piece, king);
		else if(piece.getClass() == Queen.class)
			return queenThreatens(piece, king);
		else if(piece.getClass() == King.class)
			return kingThreatens(piece, king);
		else
			return false;
	}
	
	/**
	 * Tests whether a given pawn threatens the given king.
	 * 
	 * @param pawn the pawn to test
	 * @param king the king
	 * @return true if the pawn threatens the king, false otherwise
	 */
	private final boolean pawnThreatens(Piece pawn, King king) {
		int forward = pawn.player == Player.WHITE ? 1 : -1;
		return pawn.rank + forward == king.rank && (pawn.file + 1 == king.file || pawn.file - 1 == king.file);
	}
	
	/**
	 * Tests whether a given knight threatens the given king.
	 * 
	 * @param knight the knight to test
	 * @param king the king
	 * @return true if the knight threatens the king, false otherwise
	 */
	private final boolean knightThreatens(Piece knight, King king) {
		int horizontal = horizontal(knight, king);
		int vertical = vertical(knight, king);
		if(horizontal == 0 || vertical == 0)
			return false;
		return (knight.file + horizontal == king.file && knight.rank + vertical * 2 == king.rank) ||
				(knight.file + horizontal * 2 == king.file && knight.rank + vertical == king.rank);
	}

	/**
	 * Tests whether a given bishop threatens the given king.
	 * 
	 * @param bishop the bishop to test
	 * @param king the king
	 * @return true if the bishop threatens the king, false otherwise
	 */
	private final boolean bishopThreatens(Piece bishop, King king) {
		if(bishop.file == king.file || bishop.rank == king.rank)
			return false;
		return lineTo(bishop, horizontal(bishop, king), vertical(bishop, king), king);
	}

	/**
	 * Tests whether a given rook threatens the given king.
	 * 
	 * @param rook the rook to test
	 * @param king the king
	 * @return true if the rook threatens the king, false otherwise
	 */
	private final boolean rookThreatens(Piece rook, King king) {
		if(rook.file == king.file)
			return lineTo(rook, 0, vertical(rook, king), king);
		else if(rook.rank == king.rank)
			return lineTo(rook, horizontal(rook, king), 0, king);
		else
			return false;
	}

	/**
	 * Tests whether a given queen threatens the given king.
	 * 
	 * @param queen the queen to test
	 * @param king the king
	 * @return true if the queen threatens the king, false otherwise
	 */
	private final boolean queenThreatens(Piece queen, King king) {
		return rookThreatens(queen, king) || bishopThreatens(queen, king);
	}
	
	/**
	 * Tests whether two kings threaten each other.
	 * 
	 * @param k1 the first king
	 * @param k2 the second king
	 * @return true if the kings threaten each other, false otherwise
	 */
	private final boolean kingThreatens(Piece k1, King k2) {
		return Math.abs(k1.file - k2.file) <= 1 && Math.abs(k1.rank - k2.rank) <= 1;
	}
	
	/**
	 * Returns the unit file direction that a given piece would need to travel
	 * in to reach the given king.
	 * 
	 * @param piece the piece to move
	 * @param king the king to move toward
	 * @return 1 if the king is to the right, -1 if to the left, or 0 if the king is on the same file
	 */
	private static final int horizontal(Piece piece, King king) {
		if(piece.file > king.file)
			return -1;
		else if(piece.file < king.file)
			return 1;
		else
			return 0;
	}
	
	/**
	 * Returns the unit rank direction that a given piece would need to travel
	 * in to reach the given king.
	 * 
	 * @param piece the piece to move
	 * @param king the king to move toward
	 * @return 1 if the king is above, -1 if below, or 0 if the king is on the same rank
	 */
	private static final int vertical(Piece piece, King king) {
		if(piece.rank > king.rank)
			return -1;
		else if(piece.rank < king.rank)
			return 1;
		else
			return 0;
	}
	
	/**
	 * Tests if a given piece can move along a line to reach a given king.
	 * 
	 * @param piece the piece to move
	 * @param dfile the unit change in file at each step
	 * @param drank the unit change in rank at each step
	 * @param king the king to move toward
	 * @return true if the king lies along the line, false otherwise
	 */
	private final boolean lineTo(Piece piece, int dfile, int drank, King king) {
		int file, rank;
		for(int i=1; i<8; i++) {
			file = piece.file + dfile * i;
			rank = piece.rank + drank * i;
			Piece p = getPieceAt(file, rank);
			if(p == null)
				continue;
			else if(p == king)
				return true;
			else
				return false;
		}
		return false;
	}
}
