package edu.uky.ai.chess.gui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Arrays;

import javax.swing.JPanel;

import edu.uky.ai.chess.Settings;
import edu.uky.ai.chess.state.PGN;
import edu.uky.ai.chess.state.State;

/**
 * A graphical representation of a chess board.
 * 
 * @author Stephen G. Ware
 */
public class Board extends JPanel {

	/** Version UID */
	private static final long serialVersionUID = Settings.VERSION_UID;
	
	/** The color of the board's white squares */
	private static final Color WHITE_SQUARES = Color.WHITE;
	
	/** The color of the board's black squares */
	private static final Color BLACK_SQUARES = Color.GRAY;
	
	/** The color of the square containing the active piece */
	private static final Color ACTIVE_SQUARE = Color.GREEN;
	
	/** The color of squares the active piece may move to */
	private static final Color TARGET_SQUARES = Color.RED;

	/** The current board state */
	private State current;
	
	/** The currently active piece (the one to be moved) */
	private edu.uky.ai.chess.state.Piece active = null;
	
	/** An array indicating which squares the currently active piece may move to */
	private State[][] target = new State[8][8];
	
	/**
	 * Constructs a new board in the starting configuration for a chess game.
	 */
	public Board() {
		super(true);
		this.current = new State();
		setPreferredSize(new Dimension(Piece.WIDTH * 8, Piece.HEIGHT * 8));
		addMouseListener(input);
	}
	
	/**
	 * Returns the current game state that is being visualized.
	 * 
	 * @return the current state
	 */
	public synchronized State getState() {
		return current;
	}

	/**
	 * Sets the current state to be visualized and updates the screen.
	 * 
	 * @param state the state to visualize
	 */
	public synchronized void setState(State state) {
		this.current = state;
		setActive(null);
		repaint();
	}
	
	/**
	 * Sets the active piece.  The active piece is one which a human player is
	 * considering moves for.
	 * 
	 * @param piece the piece to be made active (or null to unset the active piece)
	 */
	public synchronized void setActive(edu.uky.ai.chess.state.Piece piece) {
		for(State[] t : target)
			Arrays.fill(t, null);
		active = piece;
		if(active != null) {
			for(State next : current.next(piece)) {
				edu.uky.ai.chess.state.Piece to = PGN.getNewLocation(current.board, next.board, active);
				target[to.file][to.rank] = next;
			}
		}
		repaint();
	}
	
	@Override
	public void paintComponent(Graphics g) {
		State state = getState();
		Graphics2D g2d = (Graphics2D) g;
		for(int file = 0; file < 8; file++) {
			for(int rank = 0; rank < 8; rank++) {
				if(file % 2 == 0) {
					if(rank % 2 == 0)
						g2d.setColor(BLACK_SQUARES);
					else
						g2d.setColor(WHITE_SQUARES);
				}
				else {
					if(rank % 2 == 0)
						g2d.setColor(WHITE_SQUARES);
					else
						g2d.setColor(BLACK_SQUARES);
				}
				if(active != null) {
					if(active.file == file && active.rank == rank)
						g2d.setColor(ACTIVE_SQUARE);
					else if(target[file][rank] != null)
						g2d.setColor(TARGET_SQUARES);
				}
				int x = file * Piece.WIDTH;
				int y = (Piece.HEIGHT * 7) - Piece.HEIGHT * rank;
				g2d.fillRect(x, y, Piece.WIDTH, Piece.HEIGHT);
				edu.uky.ai.chess.state.Piece piece = state.board.getPieceAt(file, rank);
				if(piece != null)
					g2d.drawImage(Piece.get(piece), x, y, null);
			}
		}
	}
	
	/**
	 * A object used to wait for a human user to input a move.
	 * 
	 * @author Stephen G. Ware
	 */
	private final class MoveRequest {
		
		/** The next state chosen by the user (initially null) */
		public State move = null;
		
		/**
		 * Causes this object to wait until it is notified (at which point
		 * {@link #move} should be set.
		 * 
		 * @return the next state that was chosen
		 */
		public synchronized State waitForMove() {
			try {
				wait();
			}
			catch(InterruptedException e) {
				// do nothing
			}
			return move;
		}
		
		/**
		 * Sets the next state to return.
		 * 
		 * @param move the next state chosen by the user
		 */
		public synchronized void answer(State move) {
			this.move = move;
			notify();
		}
	}
	
	/** The currently active request (if any) */
	private MoveRequest request = null;
	
	/**
	 * Allows a human user to input a move by clicking on a piece and choosing
	 * its destination.  This method blocks until the user has finished
	 * inputting a move.
	 * 
	 * @return the next state input by the user
	 */
	public State request() {
		if(request != null)
			throw new IllegalStateException("Another move request is already pending.");
		request = new MoveRequest();
		State move = request.waitForMove();
		if(move == null)
			throw new IllegalStateException("Move request never satisfied.");
		request = null;
		return move;
	}
	
	/** Used to collect mouse input from the user */
	private final MouseAdapter input = new MouseAdapter() {
		
		@Override
		public void mouseClicked(MouseEvent event) {
			if(request == null)
				return;
			int file = event.getX() / Piece.WIDTH;
			int rank = (8 * Piece.HEIGHT - event.getY()) / Piece.HEIGHT;
			edu.uky.ai.chess.state.Piece piece = current.board.getPieceAt(file, rank);
			if(active == null && piece != null && piece.player == current.player)
				setActive(piece);
			else if(active == piece)
				setActive(null);
			else if(target[file][rank] != null)
				request.answer(target[file][rank]);
		}
	};
}
