package edu.uky.ai.chess;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.function.Function;

import edu.uky.ai.SearchBudget;
import edu.uky.ai.chess.gui.Piece;
import edu.uky.ai.io.DummyWriter;
import edu.uky.ai.util.Arguments;
import edu.uky.ai.util.Table;
import edu.uky.ai.util.Utilities;
import edu.uky.ai.util.Table.Sequence;

/**
 * The entry point for the chess application.
 * 
 * @author Stephen G. Ware
 */
public class Main {
	
	private static final String USAGE = "Usage: java -jar chess.jar -a <agents> -g <games> [-nl <moves>] [-tl <millis>] [-o <file>] [-b]" +
		"\n  <agents> is two or more JAR files containing an instance of " + Agent.class.getName() +
		"\n  <games>  is the number of matches to play between each pair of agents" +
		"\n  <moves>  is the optional maximum number of moves an agent may consider when making a decision" +
		"\n  <millis> is the optional maximum number of milliseconds an agent may take to decide on a move" +
		"\n  <file>   is the file to which output will be written in HTML format" +
		"\n  -b       indicates batch mode, that the program should exit immediately when finished";

	/**
	 * Launches the chess program according to its command line arguments.
	 * 
	 * @param args the command line arguments
	 * @throws Exception if any uncaught exceptions are thrown
	 */
	public static void main(String[] args) throws Exception {
		if(args.length < 5) {
			System.out.println(USAGE);
			System.exit(1);
		}
		Arguments arguments = new Arguments(args);
		ArrayList<Agent> agents = new ArrayList<>();
		for(String url : arguments.getValues("-a"))
			agents.add(Utilities.loadFromJARFile(Agent.class, new File(url)));
		int games = Integer.parseInt(arguments.getValue("-g"));
		int maxMoves = SearchBudget.INFINITE_OPERATIONS;
		if(arguments.containsKey("-nl"))
			maxMoves = Integer.parseInt(arguments.getValue("-nl"));
		long maxTime = SearchBudget.INFINITE_TIME;
		if(arguments.containsKey("-tl"))
			maxTime = Long.parseLong(arguments.getValue("-tl"));
		final Writer output;
		if(arguments.containsKey("-o"))
			output = new BufferedWriter(new FileWriter(arguments.getValue("-o")));
		else
			output = new DummyWriter();
		try {
			play(agents.toArray(new Agent[agents.size()]), games, maxMoves, maxTime, output);
		}
		finally {
			output.close();
		}
		if(arguments.containsKey("-b"))
			System.exit(0);
	}
	
	/**
	 * Players a tournament between the given chess agents.
	 * 
	 * @param agents the chess agents
	 * @param numberOfGames the number of games each ordered pair of players
	 * will play
	 * @param maxMoves the maximum moves that may be considered when an agent
	 * makes a decision
	 * @param maxTime the maximum milliseconds an agent may take to decide on a
	 * move
	 * @param output where the results will be written (as HTML)
	 * @throws IOException if a problem occurs while writing to the output
	 */
	public static void play(Agent[] agents, int numberOfGames, int maxMoves, long maxTime, Writer output) throws IOException {
		Piece.load();
		Table results = new Table(agents, agents);
		ArrayList<Game> games = new ArrayList<>();
		for(Agent row : agents)
			for(Agent column : agents)
				if(row != column)
					results.getCell(row, column).value = new Result(row, column);
		for(int g=0; g<numberOfGames; g++) {
			for(Agent white : agents) {
				for(Agent black : agents) {
					if(white != black) {
						Game game = new Game(white, black, maxMoves, maxTime);
						game.play();
						games.add(game);
						((Result) results.getCell(white, black).value).add(game);
						((Result) results.getCell(black, white).value).add(game);
					}
				}
			}
		}
		results = results.sort(BEST_AGENTS, BEST_AGENTS.reversed());
		Table score = results.transform(SCORE).addTotalRow().addTotalColumn();
		System.out.println("Results:\n" + score);
		output.append("<html>\n<head>\n<title>Chess Tournament Results</title>");
		output.append("\n<style>\ntable { border-collapse: collapse; }\ntable, tr, th, td { border: 1px solid black; }\ntr:nth-child(odd) { background-color: lightgray; }\nth { font-weight: bold; }\ntd { text-align: right; }\n</style>");
		output.append("\n</head>\n<body>\n\n<h1>Chess Tournament Results</h1>");
		output.append("\n\n<h2>Points Scored</h2>\n" + score.toHTML());
		output.append("\n\n<h2>Moves Made</h2>\n" + results.transform(MOVES).addTotalRow().addTotalColumn().toHTML());
		output.append("\n\n<h2>Moves Considered</h2>\n" + results.transform(STATES).addTotalRow().addTotalColumn().toHTML());
		for(int i=0; i<games.size(); i++)
			output.append("\n\n<h2>Game " + (i + 1) + "</h2>\n<pre>\n" + games.get(i) + "\n</pre>");
		output.append("\n\n</body>\n<html>");
		output.flush();
	}
	
	private static final Comparator<Table.Sequence> BEST_AGENTS = new Comparator<Table.Sequence>() {
		@Override
		public int compare(Sequence sequence1, Sequence sequence2) {
			Number difference = Utilities.subtract(sequence2.sum(SCORE), sequence1.sum(SCORE));
			if(difference.doubleValue() == 0d)
				difference = Utilities.subtract(sequence1.sum(MOVES), sequence2.sum(MOVES));
			if(difference.doubleValue() == 0d)
				difference = Utilities.subtract(sequence1.sum(STATES), sequence2.sum(STATES));
			if(difference.doubleValue() == 0d)
				return 0;
			else if(difference.doubleValue() < 0d)
				return -1;
			else
				return 1;
		}
	};
	
	private static final Function<Object, Object> SCORE = object -> object instanceof Result ? ((Result) object).score : object == null ? "-" : object;
	
	private static final Function<Object, Object> MOVES = object -> object instanceof Result ? ((Result) object).moves : object == null ? "-" : object;
	
	private static final Function<Object, Object> STATES = object -> object instanceof Result ? ((Result) object).states : object == null ? "-" : object;
}
