package edu.uky.ai.chess.state;

import java.util.function.Predicate;

/**
 * Represents an individual piece of one color at a given location.  This is
 * the abstract superclass of all chess pieces.  Note that it includes the
 * pieces current position, so this object represents a given piece at a given
 * position.  When a piece is moved, a new piece object is created with the new
 * position information.
 * 
 * @author Stephen G. Ware
 */
public abstract class Piece {

	/** The player who controls this piece */
	public final Player player;
	
	/** The piece's file (column) on the board (A to H), as an integer (0 to 7, where 0=A, 1=B, etc.) */
	public final int file;
	
	/** The piece's rank (row) on the board (1 to 8), as an integer (0 to 7, where 0=1, 1=2, etc.) */
	public final int rank;
	
	/**
	 * Constructs a new piece for a given player at a given location.
	 * 
	 * @param player the player who controls the piece
	 * @param file the piece's file (column) on the board
	 * @param rank the piece's rank (row) on the board
	 */
	public Piece(Player player, int file, int rank) {
		this.player = player;
		this.file = file;
		this.rank = rank;
	}
	
	@Override
	public boolean equals(Object other) {
		if(getClass() == other.getClass()) {
			Piece otherPiece = (Piece) other;
			return player == otherPiece.player && file == otherPiece.file && rank == otherPiece.rank;
		}
		return false;
	}
	
	@Override
	public int hashCode() {
		return toString().hashCode();
	}
	
	/**
	 * Given a board state and a predicate, this method will test every next
	 * board state that can result from one move of this piece.  If the
	 * predicate ever returns true, no additional board states will be tried
	 * and this method will return true.
	 * 
	 * @param board the board state before a move is made
	 * @param predicate a predicate for testing board states
	 * @return true if the predicate ever returned true, false otherwise
	 */
	abstract boolean anyMove(Board board, Predicate<Board> predicate);
	
	/**
	 * Returns a new piece object of the same type which has been moved a given
	 * number of squares on the board.
	 * 
	 * @param dfile the number of squares to move horizontally (positive for right, negative for left)
	 * @param drank the number of squares to move vertically (positive to up, negative for down)
	 * @return a piece of the same type at the new location
	 */
	public abstract Piece move(int dfile, int drank);
}
