package edu.uky.ai.chess.state;

import java.util.ArrayList;
import java.util.HashSet;

/**
 * Provides methods for generating Portable Game Notation from a sequence of
 * chess states.
 * 
 * @author Stephen G. Ware
 */
public class PGN {
	
	/**
	 * Returns the file letter mapped to a given index.
	 * 
	 * @param file the file index
	 * @return the file letter
	 */
	public static String file(int file) {
		switch(file){
		case 0: return "a";
		case 1: return "b";
		case 2: return "c";
		case 3: return "d";
		case 4: return "e";
		case 5: return "f";
		case 6: return "g";
		case 7: return "h";
		default: return Integer.toString(file);
		}
	}
	
	/**
	 * Returns the rank number mapped to a given index.
	 * 
	 * @param rank the rank index
	 * @return the rank number
	 */
	public static String rank(int rank) {
		return Integer.toString(rank + 1);
	}
	
	/**
	 * Returns the alphanumeric representation of a square on a chess board
	 * given its file and rank indices.
	 * 
	 * @param file the file index
	 * @param rank the rank index
	 * @return the algebraic notation for that square (e.g. "a1")
	 */
	public static String square(int file, int rank) {
		return file(file) + rank(rank);
	}
	
	/**
	 * Returns PGN text describing the move which caused the game to transition
	 * from one state to another.
	 * 
	 * @param before the state of the game before the move
	 * @param after the state of the game after the move
	 * @return a PGN string describing the move
	 */
	public static String move(State before, State after) {
		Piece from = first(difference(after.board, before.board, before.player));
		Piece to = first(difference(before.board, after.board, before.player));
		int dfile = to.file - from.file;
		if(from.getClass() == King.class && Math.abs(dfile) == 2) {
			if(dfile == 2)
				return "O-O";
			else
				return "O-O-O";
		}
		String str = from.getClass() == Pawn.class ? "" : from.toString();
		str += disambiguate(before, from, after, to);
		if(before.board.countPieces() > after.board.countPieces())
			str += "x";
		str += file(to.file) + rank(to.rank);
		if(from.getClass() != to.getClass())
			str += "=" + to.toString();
		if(after.check) {
			if(after.outcome == Outcome.WHITE_WINS || after.outcome == Outcome.BLACK_WINS)
				str += "#";
			else
				str += "+";
		}
		return str;
	}
	
	/**
	 * Returns the PGN text describing the game up to the point represented by
	 * a given state object.
	 * 
	 * @param state the current state of the game
	 * @return a PGN string describing the game
	 */
	public static String game(State state) {
		String string = "";
		while(state.previous != null) {
			if(state.player == Player.WHITE)
				string = " " + move(state.previous, state) + "\n" + string;
			else
				string = (state.turn + 1) + ". " + move(state.previous, state) + string;
			state = state.previous;
		}
		return string;
	}
	
	/**
	 * Given the state of a chess board before and after a move and a piece to
	 * track, this method returns the piece at the location it occupied before
	 * the move.
	 * 
	 * @param before the state of the board before the move
	 * @param after the state of the board after the move
	 * @param piece a piece of the after board
	 * @return the corresponding piece on the before board
	 */
	public static Piece getNewLocation(Board before, Board after, Piece piece) {
		HashSet<Piece> difference = difference(before, after, piece.player);
		for(Piece p : difference)
			if(p.getClass() == piece.getClass())
				return p;
		for(Piece p : difference)
			if(p.player == piece.player)
				return p;
		return piece;
	}

	/**
	 * Returns the pieces which appear on the <tt>after</tt> board but not on
	 * the <tt>before</tt>board.
	 * 
	 * @param before the state of the board before the move
	 * @param after the state of the board after the move
	 * @param player the player whose pieces should be checked
	 * @return a set of pieces not present on the before board
	 */
	private static HashSet<Piece> difference(Board before, Board after, Player player) {
		HashSet<Piece> pieces = new HashSet<>();
		for(Piece p : after)
			if(p.player == player)
				pieces.add(p);
		for(Piece p : before)
			if(p.player == player)
				pieces.remove(p);
		return pieces;
	}
	
	/**
	 * Returns the first element of an {java.lang.Iterable}.
	 * 
	 * @param pieces the iterable
	 * @return the first element, or null if it is empty
	 */
	private static Piece first(Iterable<Piece> pieces) {
		for(Piece piece : pieces)
			return piece;
		return null;
	}
	
	/**
	 * Given a piece to move, this method returns the string (possibly empty)
	 * that is needed to disambiguate this piece from other pieces of the same
	 * type which might have moved to the same location.
	 * 
	 * @param before the state of the board before the move
	 * @param from the piece on the board before the move
	 * @param after the state of the board after the move
	 * @param to the piece on the board after the move
	 * @return the string needed to disambiguate the piece from others
	 */
	private static final String disambiguate(State before, Piece from, State after, Piece to) {
		ArrayList<Piece> ambiguous = new ArrayList<>();
		Board beforeClone = before.board.clone();
		before.board.anyMove(to.player, (next) -> {
			if(next.inCheck(to.player))
				return false;
			if(next.pieceAt(to.file, to.rank, to.player, to.getClass())) {
				Piece p = first(difference(next, beforeClone, to.player));
				if(p != null)
					ambiguous.add(p);
			}
			return false;
		});
		boolean rankNeeded = false;
		for(int i=0; i<ambiguous.size() - 1; i++)
			for(int j=i+1; j<ambiguous.size(); j++)
				if(ambiguous.get(i).file == ambiguous.get(j).file)
					rankNeeded = true;
		String str = "";
		if(ambiguous.size() > 0 || (from.getClass() == Pawn.class && from.file != to.file))
			str += file(from.file);
		if(rankNeeded)
			str += rank(from.rank);
		return str;
	}
}
