package edu.uky.ai.chess.gui;

import java.awt.image.BufferedImage;
import java.io.IOException;

import javax.imageio.ImageIO;

import edu.uky.ai.chess.state.*;

/**
 * Represents an individual chess piece as a GUI element.
 * 
 * @author Stephen G. Ware
 */
public class Piece {

	/** The width of a piece's image in pixels */
	static final int WIDTH = 60;
	
	/** The height of a piece's image in pixels */
	static final int HEIGHT = 60;
	
	/** Index of the pawn image in the spritesheet and in the image arrays */
	private static final int PAWN = 0;
	
	/** Index of the bishop image in the spritesheet and in the image arrays */
	private static final int BISHOP = 1;
	
	/** Index of the knight image in the spritesheet and in the image arrays */
	private static final int KNIGHT = 2;
	
	/** Index of the rook image in the spritesheet and in the image arrays */
	private static final int ROOK = 3;
	
	/** Index of the queen image in the spritesheet and in the image arrays */
	private static final int QUEEN = 4;
	
	/** Index of the king image in the spritesheet and in the image arrays */
	private static final int KING = 5;
	
	/** Images for the white pieces */
	private static final BufferedImage white[] = new BufferedImage[6];
	
	/** Images for the black pieces */
	private static final BufferedImage black[] = new BufferedImage[6];
	
	/**
	 * Reads the spritesheet (which should be a file named 'pieces.png' in this
	 * package) and breaks it into smaller images, one for each chess piece of
	 * each color.
	 * 
	 * @throws IOException if an error occurs while reading the image
	 */
	public static final void load() throws IOException {
		String url = Piece.class.getPackage().getName();
		url = "/" + url.replace('.', '/') + "/pieces.png";
		BufferedImage all = ImageIO.read(Piece.class.getResource(url));
		copyPieces(all, white, 0);
		copyPieces(all, black, HEIGHT);
	}
	
	/**
	 * Given the loaded spritesheet and an array of smaller images, this method
	 * breaks one horizontal row of the larger image into smaller parts and
	 * places them into the given array.
	 * 
	 * @param all the spritesheet image
	 * @param pieces the array into which the smaller images will be placed
	 * @param y the row of smaller images to be read
	 */
	private static final void copyPieces(BufferedImage all, BufferedImage[] pieces, int y) {
		for(int i=0; i<6; i++)
			pieces[i] = all.getSubimage(i * WIDTH, y, WIDTH, HEIGHT);
	}
	
	/**
	 * Given a chess piece, this method returns the image associated with that
	 * piece.
	 * 
	 * @param piece the piece to visualize
	 * @return the image representing that piece
	 */
	static final BufferedImage get(edu.uky.ai.chess.state.Piece piece) {
		BufferedImage[] color;
		if(piece.player == Player.WHITE)
			color = white;
		else
			color = black;
		BufferedImage image;
		if(piece instanceof Knight)
			image = color[KNIGHT];
		else if(piece instanceof Bishop)
			image = color[BISHOP];
		else if(piece instanceof Rook)
			image = color[ROOK];
		else if(piece instanceof Queen)
			image = color[QUEEN];
		else if(piece instanceof King)
			image = color[KING];
		else
			image = color[PAWN];
		return image;
	}
}
