package edu.uky.ai.chess;

import edu.uky.ai.SearchBudget;
import edu.uky.ai.chess.state.State;

/**
 * The abstract parent class of all chess playing agents.
 * 
 * @author Stephen G. Ware
 */
public abstract class Agent {

	/** The agent's name */
	public final String name;
	
	/**
	 * Constructs a new agent with the given name.
	 * 
	 * @param name the agent's name
	 */
	public Agent(String name) {
		this.name = name;
	}
	
	@Override
	public String toString() {
		return name;
	}
	
	/**
	 * Given the current state of a chess game, this method chooses the next
	 * move for current player. This method should return one of the children
	 * of state object it is passed. If it returns anything else, the current
	 * player will concede.
	 * 
	 * @param current the current state of the game
	 * @param maxMoves the maximum moves that may be considered when an agent
	 * makes a decision
	 * @param maxTime the maximum milliseconds an agent may take to decide on a
	 * move
	 * @return a child of the state object, which represents the move to make,
	 * or null to concede
	 * @throws IllegalStateException if the agent returns an invalid move
	 * (null, the current state, a state which is not the child of the current
	 * state, a state which uses a different search budget, etc.)
	 */
	public final State choose(State current, int maxMoves, long maxTime) {
		SearchBudget budget = new SearchBudget(maxMoves, maxTime);
		State limited = current.setSearchBudet(budget);
		State choice = chooseMove(limited);
		if(choice == null)
			throw new IllegalStateException("Agent returned null.");
		else if(choice == current)
			throw new IllegalStateException("Agent returned the current state.");
		else if(choice.budget != budget)
			throw new IllegalStateException("The wrong search budget was used.");
		else if(choice.previous != limited)
			throw new IllegalStateException("Invalid move: " + choice + " (not a child of the current state)");
		return choice;
	}
	
	/**
	 * Given the current {@link edu.uky.ai.chess.state.State}, this method
	 * chooses a next move for the current player. The color of the current
	 * player can be checked using {@link edu.uky.ai.chess.state.State#player}.
	 * This method must return a child of the state node it is passed. If this
	 * method does anything else (e.g. returns a state node that is not a
	 * direct child of the state node that was passed, returns null, or throws
	 * an exception) the current agent will concede the game.
	 * 
	 * @param current the current state of the game
	 * @return a child of the state object, which represents the move to make,
	 * or null to concede the game
	 */
	protected abstract State chooseMove(State current);
}
