package edu.uky.ai.rl.dungeon;

import java.io.Serializable;
import java.util.Random;

import edu.uky.ai.Settings;

abstract class Tile implements Serializable {

	private static final long serialVersionUID = Settings.VERSION_UID;
	static final String UNKNOWN = "?";
	static final String ENTERANCE = "E";
	static final String EXIT = "X";
	static final String EMPTY = " ";
	static final String WALL = "W";
	static final String RAT_NAME = "rat";
	static final String ALIVE_RAT = "R";
	static final String DEAD_RAT = "r";
	static final String GOBLIN_NAME = "goblin";
	static final String ALIVE_GOBLIN = "G";
	static final String DEAD_GOBLIN = "g";
	static final String DRAGON_NAME = "dragon";
	static final String ALIVE_DRAGON = "D";
	static final String DEAD_DRAGON = "d";
	static final String CLOSED_CHEST = "C";
	static final String OPEN_CHEST = "c";
	static final String ALIVE_PLAYER = "P";
	static final String DEAD_PLAYER = "p";
	
	final int x;
	final int y;
	
	Tile(int x, int y) {
		this.x = x;
		this.y = y;
	}
	
	@Override
	public abstract String toString();
	
	String toString(DungeonState state) {
		return toString();
	}
	
	final DungeonTransition moveTo(DungeonState state, Random random) {
		if(state.terminal)
			throw new IllegalStateException("Cannot transition past a terminal state.");
		else
			return move(state, random);
	}
	
	abstract DungeonTransition move(DungeonState state, Random random);
	
	private static abstract class Passable extends Tile {

		private static final long serialVersionUID = Settings.VERSION_UID;
		
		Passable(int x, int y) {
			super(x, y);
		}
		
		@Override
		String toString(DungeonState state) {
			if(state.x == x && state.y == y) {
				if(state.terminal)
					return DEAD_PLAYER;
				else
					return ALIVE_PLAYER;
			}
			else
				return toString();
		}
	}
	
	static final class Enterance extends Passable {

		private static final long serialVersionUID = Settings.VERSION_UID;
		
		Enterance(int x, int y) {
			super(x, y);
		}

		@Override
		public String toString() {
			return ENTERANCE;
		}
		
		@Override
		DungeonTransition move(DungeonState state, Random random) {
			return new DungeonTransition(state.moveTo(x, y), -1, "You take a step.");
		}
	}
	
	static final class Exit extends Passable {

		private static final long serialVersionUID = Settings.VERSION_UID;
		
		Exit(int x, int y) {
			super(x, y);
		}

		@Override
		public String toString() {
			return EXIT;
		}
		
		@Override
		String toString(DungeonState state) {
			if(state.x == x && state.y == y)
				return ALIVE_PLAYER;
			else
				return toString();
		}
		
		@Override
		DungeonTransition move(DungeonState state, Random random) {
			double reward = 0;
			for(int value : state.tiles)
				reward += value;
			return new DungeonTransition(state.moveTo(x, y).terminal(), reward, "You escape the dungeon with " + reward + " gold!");
		}
	}
	
	static final class Empty extends Passable {
		
		private static final long serialVersionUID = Settings.VERSION_UID;

		Empty(int x, int y) {
			super(x, y);
		}

		@Override
		public String toString() {
			return EMPTY;
		}

		@Override
		DungeonTransition move(DungeonState state, Random random) {
			return new DungeonTransition(state.moveTo(x, y), -1, "You take a step.");
		}
	}
	
	static final class Wall extends Tile {

		private static final long serialVersionUID = Settings.VERSION_UID;

		Wall(int x, int y) {
			super(x, y);
		}

		@Override
		public String toString() {
			return WALL;
		}

		@Override
		DungeonTransition move(DungeonState state, Random random) {
			return new DungeonTransition(state, -1, "You bump into a wall.");
		}
	}
	
	static abstract class StatefulTile extends Tile {
		
		private static final long serialVersionUID = Settings.VERSION_UID;
		final int index;
		final double difficulty;
		final int value;
		
		StatefulTile(int x, int y, int index, double difficulty, int value) {
			super(x, y);
			this.index = index;
			this.difficulty = difficulty;
			this.value = value;
		}
	}
	
	static final class Monster extends StatefulTile {

		private static final long serialVersionUID = Settings.VERSION_UID;
		final String name;
		final String alive;
		final String dead;
		
		Monster(int x, int y, int index, double difficulty, int value, String name, String alive, String dead) {
			super(x, y, index, difficulty, value);
			this.name = name;
			this.alive = alive;
			this.dead = dead;
		}

		@Override
		public String toString() {
			return alive;
		}
		
		@Override
		String toString(DungeonState state) {
			if(state.x == x && state.y == y) {
				if(state.terminal)
					return DEAD_PLAYER;
				else
					return ALIVE_PLAYER;
			}
			if(state.tiles.get(index) > 0)
				return dead;
			else
				return alive;
		}

		@Override
		DungeonTransition move(DungeonState state, Random random) {
			if(state.tiles.get(index) > 0)
				return new DungeonTransition(state.moveTo(x, y), -1, "You tread on the remains of the " + name + ".");
			else if(random.nextDouble() >= difficulty)
				return new DungeonTransition(state.moveTo(x, y).set(index, value), 0, "You slay the " + name + " and loot " + value + " gold!");
			else
				return new DungeonTransition(state.terminal(), -1, "You are killed by the " + name + "!");
		}
	}
	
	static final class Chest extends StatefulTile {

		private static final long serialVersionUID = Settings.VERSION_UID;
		
		Chest(int x, int y, int index, double difficulty, int value) {
			super(x, y, index, difficulty, value);
		}

		@Override
		public String toString() {
			return CLOSED_CHEST;
		}
		
		@Override
		String toString(DungeonState state) {
			if(state.tiles.get(index) > 0)
				return OPEN_CHEST;
			else
				return CLOSED_CHEST;
		}

		@Override
		DungeonTransition move(DungeonState state, Random random) {
			if(state.tiles.get(index) > 0)
				return new DungeonTransition(state, -1, "You've already looted that chest.");
			else if(random.nextDouble() >= difficulty)
				return new DungeonTransition(state.set(index, value), 0, "You pick the lock and find " + value + " gold inside!");
			else
				return new DungeonTransition(state, -1, "You fail to pick the lock!");
		}
	}
}
