package edu.uky.ai.rl;

import edu.uky.ai.util.Utilities;

/**
 * Represents the results of a {@link Learner} learning a {@link Policy} for a
 * {@link StochasticProcess}.
 * 
 * @author Stephen G. Ware
 */
public class Result {

	/** The learner */
	public final Learner learner;
	
	/** The stochastic process being learned */
	public final StochasticProcess process;
	
	/** Whether or not a policy was learned */
	public final boolean success;
	
	/** The policy learned */
	public final Policy policy;
	
	/** A brief, human-readable explanation of the results */
	public final String reason;
	
	/** The number of calls to {@link StochasticProcess#transition(State, Action)} made while learning the policy */
	public final int transitions;
	
	/** The time (in milliseconds) spent learning the policy */
	public final long time;
	
	/** The score the policy earned when evaluated */
	public final double score;
	
	/**
	 * Constructs a new result object.
	 * 
	 * @param learner the learner
	 * @param process the stochastic process
	 * @param policy the policy
	 * @param reason the explanation of results
	 * @param transitions the number of transitions during training
	 * @param time the number of milliseconds spent training
	 * @param score the score from the evaluation
	 */
	public Result(Learner learner, StochasticProcess process, Policy policy, String reason, int transitions, long time, double score) {
		this.learner = learner;
		this.process = process;
		this.success = policy != null;
		this.policy = policy;
		this.reason = reason;
		this.transitions = transitions;
		this.time = time;
		this.score = score;
	}
	
	@Override
	public String toString() {
		String str = "[" + learner + " ";
		if(success)
			str += "succeeded";
		else
			str += "failed";
		str += " on " + process.name + "; ";
		str += transitions + " transitions, " + Utilities.time(time) + ", score " + score;
		if(!success)
			str += "; " + reason;
		return str + "]";
	}
}
