package edu.uky.ai;

/**
 * Represents the limitations placed on an algorithm in terms of how many
 * operations it may perform and how long it make take to solve a problem.
 * 
 * @author Stephen G. Ware
 */
public final class SearchBudget {
	
	/** Represents no limit on the number of operations */
	public static final int INFINITE_OPERATIONS = -1;
	
	/** Represents no limit on the amount of time */
	public static final long INFINITE_TIME = -1;

	/** A search budget with no limitations */
	public static final SearchBudget INFINITE = new SearchBudget(INFINITE_OPERATIONS, INFINITE_TIME);
	
	/** The number of operations executed so far */
	private int operations = 0;
	
	/** The maximum number of operations which may be executed */
	private int maxOperations;
	
	/** The time at which this search budget was created */
	private long startTime = System.currentTimeMillis();
	
	/** The time at which time will have run out */
	private long endTime;
	
	/**
	 * Constructs a new search budget that allows the given number of
	 * operations and the give amount of time.
	 * 
	 * @param operations the maximum number of operations
	 * @param time the number of milliseconds that may elapse
	 */
	public SearchBudget(int operations, long time) {
		if(operations < 0 && operations != INFINITE_OPERATIONS)
			throw new IllegalArgumentException("Search operations must be positive.");
		if(time < 0 && time != INFINITE_TIME)
			throw new IllegalArgumentException("Time must be positive.");
		maxOperations = operations;
		if(time == INFINITE_TIME)
			endTime = INFINITE_TIME;
		else
			endTime = startTime + time;
	}
	
	@Override
	public String toString() {
		return "[budget: " + (maxOperations == INFINITE_OPERATIONS ? "infinite" : getRemainingOperations()) + " operations, " + (endTime == INFINITE_TIME ? "infinite" : getRemainingTime()) + " time (ms) remaining]";
	}
	
	/**
	 * Returns whether or not it is possible for this budget to run out.
	 * 
	 * @return true if the number of operations or time is finite, false if
	 * both are infinite
	 */
	public boolean isInfinite() {
		return maxOperations == INFINITE_OPERATIONS && endTime == INFINITE_TIME;
	}
	
	/**
	 * Returns true if there are no more operations or time remaining.
	 * 
	 * @return true if resources have been used up, false otherwise
	 */
	public boolean hasBeenExhausted() {
		return getRemainingOperations() == 0 || getRemainingTime() == 0;
	}
	
	/**
	 * Returns the number of operations executed so far.
	 * 
	 * @return the number of operations
	 */
	public int getOperations() {
		return operations;
	}
	
	/**
	 * Returns the number of operations that may still be executed.
	 * 
	 * @return the number of operations
	 */
	public int getRemainingOperations() {
		if(maxOperations == INFINITE_OPERATIONS)
			return Integer.MAX_VALUE;
		else
			return maxOperations - operations;
	}
	
	/**
	 * Returns the amount of time spent so far.
	 * 
	 * @return the number of milliseconds that has elapse since this object was
	 * created
	 */
	public long getTime() {
		return System.currentTimeMillis() - startTime;
	}
	
	/**
	 * Returns the amount of time remaining before time is up.
	 * 
	 * @return the number of milliseconds remaining before time is up
	 */
	public long getRemainingTime() {
		if(endTime == INFINITE_TIME)
			return Long.MAX_VALUE;
		else
			return Math.max(endTime - System.currentTimeMillis(), 0);
	}
	
	/**
	 * This method is called to signal that one operations has been performed.
	 * 
	 * @throws OperationsBudgetExceededException if the maximum number of
	 * operations have already been performed
	 */
	public void incrementOperations() {		
		if(getRemainingOperations() == 0)
			throw new OperationsBudgetExceededException();
		operations++;
	}
	
	/**
	 * When this method is called, it will throw an exception if the maximum
	 * amount of time has passed, otherwise it will do nothing.
	 * 
	 * @throws TimeBudgetExceededException if the maximum number of
	 * milliseconds have passed
	 */
	public void checkTime() {
		if(getRemainingTime() == 0)
			throw new TimeBudgetExceededException();
	}
}
